/*
 * Copyright (C) 2024 iopsys Software Solutions AB
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 2.1
 * as published by the Free Software Foundation
 *
 *	  Author: Suvendhu Hansa <suvendhu.hansa@iopsys.eu>
 *
 */

#include "dynamicdnsmngr.h"

/* ********** DynamicObj ********** */
DM_MAP_OBJ tDynamicObj[] = {
/* parentobj, nextobject, parameter */
{"Device.", tDeviceObjs, NULL},
{0}
};

#define DDNS_SERVICES_DEFAULT "/etc/ddnsmngr/servers"

static bool service_exists(const char *service)
{
	struct uci_section *s = NULL;
	bool exist = false;

	uci_foreach_sections("ddnsmngr", "server", s) {
		char *service_name = NULL;

		dmuci_get_value_by_section_string(s, "service", &service_name);
		if (DM_STRCMP(service_name, service) == 0) {
			exist = true;
			break;
		}
	}

	return exist;
}

/*************************************************************
* ENTRY METHOD
*************************************************************/
static int browseDynamicDNSClientInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct dm_data *curr_data = NULL;
	LIST_HEAD(dup_list);
	char *inst = NULL;

	synchronize_specific_config_sections_with_dmmap("ddnsmngr", "client", "dmmap_ddnsmngr", &dup_list);
	list_for_each_entry(curr_data, &dup_list, list) {

		inst = handle_instance(dmctx, parent_node, curr_data->dmmap_section, "client_instance", "client_alias");

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)curr_data, inst) == DM_STOP)
			break;
	}
	free_dmmap_config_dup_list(&dup_list);
	return 0;
}

static int browseDynamicDNSServerInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct dm_data *curr_data = NULL;
	LIST_HEAD(dup_list);
	char *inst = NULL;

	synchronize_specific_config_sections_with_dmmap("ddnsmngr", "server", "dmmap_ddnsmngr", &dup_list);
	list_for_each_entry(curr_data, &dup_list, list) {

		inst = handle_instance(dmctx, parent_node, curr_data->dmmap_section, "server_instance", "server_alias");

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)curr_data, inst) == DM_STOP)
			break;
	}
	free_dmmap_config_dup_list(&dup_list);
	return 0;
}

static int browseDynamicDNSClientHostnameInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct dm_data *curr_data = NULL;
	char *inst = NULL;
	LIST_HEAD(dup_list);

	char *client_sec = section_name(((struct dm_data *)prev_data)->config_section);

	synchronize_specific_config_sections_with_dmmap("ddnsmngr", "host", "dmmap_ddnsmngr", &dup_list);
	list_for_each_entry(curr_data, &dup_list, list) {
		char *host_client = NULL;

		dmuci_get_value_by_section_string(curr_data->config_section, "dm_parent", &host_client);
		if (DM_STRCMP(client_sec, host_client) != 0)
			continue;

		inst = handle_instance(dmctx, parent_node, curr_data->dmmap_section, "host_instance", "host_alias");

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)curr_data, inst) == DM_STOP)
			break;
	}

	free_dmmap_config_dup_list(&dup_list);

	return 0;
}

/*************************************************************
* ADD & DEL OBJ
*************************************************************/
static int addObjDynamicDNSClient(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct uci_section *s = NULL, *dmmap_s = NULL;
	char s_name[16];

	snprintf(s_name, sizeof(s_name), "client_%s", *instance);

	dmuci_add_section("ddnsmngr", "client", &s);
	dmuci_rename_section_by_section(s, s_name);
	dmuci_set_value_by_section(s, "enabled", "0");

	dmuci_add_section_bbfdm("dmmap_ddnsmngr", "client", &dmmap_s);
	dmuci_set_value_by_section(dmmap_s, "section_name", section_name(s));
	dmuci_set_value_by_section(dmmap_s, "client_instance", *instance);
	return 0;
}

static int delObjDynamicDNSClient(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	struct uci_section *s = NULL, *stmp = NULL;

	// Remove all hosts configured for this client
	char *client = section_name(((struct dm_data *)data)->config_section);
	uci_foreach_sections_safe("ddnsmngr", "host", stmp, s) {
		char *value = NULL;

		dmuci_get_value_by_section_string(s, "dm_parent", &value);
		if (DM_STRCMP(value, client) == 0) {
			struct uci_section *dmmap_section = NULL;
			get_dmmap_section_of_config_section("dmmap_ddnsmngr", "host", section_name(s), &dmmap_section);

			dmuci_delete_by_section(s, NULL, NULL);
			dmuci_delete_by_section(dmmap_section, NULL, NULL);
		}
	}

	// Remove the client
	dmuci_delete_by_section(((struct dm_data *)data)->config_section, NULL, NULL);
	dmuci_delete_by_section(((struct dm_data *)data)->dmmap_section, NULL, NULL);
	return 0;
}

static int addObjDynamicDNSServer(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct uci_section *s = NULL, *dmmap_s = NULL;
	char s_name[16];

	snprintf(s_name, sizeof(s_name), "server_%s", *instance);

	dmuci_add_section("ddnsmngr", "server", &s);
	dmuci_rename_section_by_section(s, s_name);
	dmuci_set_value_by_section(s, "enabled", "0");

	dmuci_add_section_bbfdm("dmmap_ddnsmngr", "server", &dmmap_s);
	dmuci_set_value_by_section(dmmap_s, "name", section_name(s));
	dmuci_set_value_by_section(dmmap_s, "server_instance", *instance);
	return 0;
}

static int delObjDynamicDNSServer(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	struct uci_section *s = NULL;
	char *service_name = NULL;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "service", &service_name);
	dmuci_delete_by_section(((struct dm_data *)data)->config_section, NULL, NULL);
	dmuci_delete_by_section(((struct dm_data *)data)->dmmap_section, NULL, NULL);

	if (DM_STRLEN(service_name) == 0)
		return 0;

	// Now need to remove the service if configured to any client
	uci_foreach_sections("ddnsmngr", "client", s) {
		char *client_service = NULL;

		dmuci_get_value_by_section_string(s, "service_name", &client_service);
		if (DM_STRCMP(service_name, client_service) == 0)
			dmuci_set_value_by_section(s, "service_name", "");
	}

	return 0;
}

static int addObjDynamicDNSClientHostname(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct uci_section *host_s = NULL, *dmmap_s = NULL;
	char buff[64];

	char *client_s = section_name(((struct dm_data *)data)->config_section);
	if (DM_STRLEN(client_s) == 0)
		return 0;

	snprintf(buff, sizeof(buff), "%s_host_%s", client_s, *instance);

	dmuci_add_section("ddnsmngr", "host", &host_s);
	dmuci_rename_section_by_section(host_s, buff);
	dmuci_set_value_by_section(host_s, "enabled", "0");
	dmuci_set_value_by_section(host_s, "dm_parent", client_s);

	dmuci_add_section_bbfdm("dmmap_ddnsmngr", "host", &dmmap_s);
	dmuci_set_value_by_section(dmmap_s, "section_name", section_name(host_s));
	dmuci_set_value_by_section(dmmap_s, "host_instance", *instance);

	return 0;
}

static int delObjDynamicDNSClientHostname(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	dmuci_delete_by_section(((struct dm_data *)data)->config_section, NULL, NULL);
	dmuci_delete_by_section(((struct dm_data *)data)->dmmap_section, NULL, NULL);
	return 0;
}

/*************************************************************
* GET & SET PARAM
*************************************************************/
static int get_DynamicDNS_ClientNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseDynamicDNSClientInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_DynamicDNS_ServerNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseDynamicDNSServerInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_DynamicDNS_SupportedServices(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_option_value_string("ddnsmngr", "global", "supported_services", value);
	return 0;
}

static int get_DynamicDNSClient_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "enabled", "0");
	return 0;
}

static int set_DynamicDNSClient_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "enabled", b ? "1" : "0");
			break;
	}
	return 0;
}

static int get_DynamicDNSClient_Status(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *res = NULL;

	char *enable = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "enabled", "0");
	if (DM_STRCMP(enable, "0") == 0) {
		*value = dmstrdup("Disabled");
		return 0;
	}

	char *client_sec = section_name(((struct dm_data *)data)->config_section);
	*value = dmstrdup("Error");

	dmubus_call("ddnsmngr", "client_status", UBUS_ARGS{{"client", client_sec, String}}, 1, &res);
	if (res != NULL) {
		*value = dmjson_get_value(res, 1, "status");
	}

	return 0;
}

static int get_DynamicDNSClient_LastError(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *res = NULL;

	*value = dmstrdup("NO_ERROR");

	char *enable = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "enabled", "0");
	if (DM_STRCMP(enable, "0") == 0) {
		return 0;
	}

	char *client_sec = section_name(((struct dm_data *)data)->config_section);
	dmubus_call("ddnsmngr", "last_error", UBUS_ARGS{{"client", client_sec, String}}, 1, &res);
	if (res != NULL) {
		*value = dmjson_get_value(res, 1, "last_error");
	}

	return 0;
}

static int get_DynamicDNSClient_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return bbf_get_alias(ctx, ((struct dm_data *)data)->dmmap_section, "client_alias", instance, value);
}

static int set_DynamicDNSClient_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	return bbf_set_alias(ctx, ((struct dm_data *)data)->dmmap_section, "client_alias", instance, value);
}

static int get_DynamicDNSClient_Server(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *service_name = NULL;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "service_name", &service_name);
	if (DM_STRLEN(service_name) == 0)
		return 0;

	return _bbfdm_get_references(ctx, "Device.DynamicDNS.Server.", "ServiceName", service_name, value);
}

static int set_DynamicDNSClient_Server(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *allowed_objects[] = {"Device.DynamicDNS.Server.", NULL};
	struct dm_reference reference = {0};

	bbfdm_get_reference_linker(ctx, value, &reference);

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, reference.path, -1, 256, NULL, NULL))
				return FAULT_9007;

			if (dm_validate_allowed_objects(ctx, &reference, allowed_objects))
				return FAULT_9007;

			if (DM_STRLEN(reference.value) == 0) {
				bbfdm_set_fault_message(ctx, "This Server object has no ServiceName.");
				return FAULT_9001;
			}

			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "service_name", reference.value);
			break;
	}
	return 0;
}

static int get_DynamicDNSClient_Interface(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *interface = NULL;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "interface", &interface);
	return _bbfdm_get_references(ctx, "Device.IP.Interface.", "Name", interface, value);
}

static int set_DynamicDNSClient_Interface(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *allowed_objects[] = {"Device.IP.Interface.", NULL};
	struct dm_reference reference = {0};

	bbfdm_get_reference_linker(ctx, value, &reference);

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, reference.path, -1, 256, NULL, NULL))
				return FAULT_9007;

			if (dm_validate_allowed_objects(ctx, &reference, allowed_objects))
				return FAULT_9007;

			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "interface", reference.value);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "ip_network", reference.value);
			break;
	}
	return 0;
}

static int get_DynamicDNSClient_Username(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "username", value);
	return 0;
}

static int set_DynamicDNSClient_Username(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 256, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "username", value);
			break;
	}
	return 0;
}

static int get_DynamicDNSClient_Password(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "password", value);
	return 0;
}

static int set_DynamicDNSClient_Password(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 256, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "password", value);
			break;
	}
	return 0;
}

static int get_DynamicDNSClient_HostnameNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseDynamicDNSClientHostnameInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_DynamicDNSClientHostname_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "enabled", "0");
	return 0;
}

static int set_DynamicDNSClientHostname_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "enabled", b ? "1" : "0");
			break;
	}
	return 0;
}

static int get_DynamicDNSClientHostname_Status(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *res = NULL;

	char *enable = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "enabled", "0");
	if (DM_STRCMP(enable, "0") == 0) {
		*value = dmstrdup("Disabled");
		return 0;
	}

	char *host_sec = section_name(((struct dm_data *)data)->config_section);
	*value = dmstrdup("Error");

	dmubus_call("ddnsmngr", "host_status", UBUS_ARGS{{"host", host_sec, String}}, 1, &res);
	if (res != NULL) {
		*value = dmjson_get_value(res, 1, "status");
	}

	return 0;
}

static int get_DynamicDNSClientHostname_Name(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "lookup_host", value);
	return 0;
}

static int set_DynamicDNSClientHostname_Name(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 256, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "lookup_host", value);
			break;
	}
	return 0;
}

static int get_DynamicDNSClientHostname_LastUpdate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *res = NULL;

	char *host_sec = section_name(((struct dm_data *)data)->config_section);
	*value = dmstrdup("0001-01-01T00:00:00Z");

	dmubus_call("ddnsmngr", "last_update", UBUS_ARGS{{"host", host_sec, String}}, 1, &res);
	if (res != NULL) {
		*value = dmjson_get_value(res, 1, "last_update");
	}

	return 0;
}

static int get_DynamicDNSServer_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "enabled", "0");
	return 0;
}

static int set_DynamicDNSServer_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "enabled", b ? "1" : "0");
			break;
	}
	return 0;
}

static int get_DynamicDNSServer_Name(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "name", value);
	return 0;
}

static int set_DynamicDNSServer_Name(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 64, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "name", value);
			break;
	}
	return 0;
}

static int get_DynamicDNSServer_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return bbf_get_alias(ctx, ((struct dm_data *)data)->dmmap_section, "server_alias", instance, value);
}

static int set_DynamicDNSServer_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	return bbf_set_alias(ctx, ((struct dm_data *)data)->dmmap_section, "server_alias", instance, value);
}

static int get_DynamicDNSServer_ServiceName(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "service", value);
	return 0;
}

static int set_DynamicDNSServer_ServiceName(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char file_path[128] = {0};
	char *curr_service = NULL;
	char *client_service = NULL;
	struct uci_section *s = NULL;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 256, NULL, NULL))
				return FAULT_9007;

			if (DM_STRLEN(value) == 0)
				break;

			// Value must be a member of SupportedServices list
			snprintf(file_path, sizeof(file_path), "/%s/%s.json", DDNS_SERVICES_DEFAULT, value);
			if (!file_exists(file_path))
				return FAULT_9007;

			// Value must not be configured with other instance since its a unique paramete
			dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "service", &curr_service);
			if (DM_STRCMP(curr_service, value) == 0)
				break; // instance with same existing value should allow

			if (service_exists(value)) {
				bbfdm_set_fault_message(ctx, "ServiceName already exist.");
				return FAULT_9001;
			}

			break;
		case VALUESET:
			dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "service", &curr_service);
			if (DM_STRCMP(curr_service, value) == 0) // Same value as existing
				break;

			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "service", value);
			if (DM_STRLEN(curr_service) == 0)
				break;

			// Now need to modify all clients configured with older service name
			uci_foreach_sections("ddnsmngr", "client", s) {
				dmuci_get_value_by_section_string(s, "service_name", &client_service);
				if (DM_STRCMP(curr_service, client_service) == 0) {
					dmuci_set_value_by_section(s, "service_name", value);
				}
			}
			break;
	}
	return 0;
}

static int get_DynamicDNSServer_SupportedProtocols(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("HTTP,HTTPS");
	return 0;
}

static int get_DynamicDNSServer_Protocol(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *service = NULL;
	char file_path[128] = {0};

	*value = dmstrdup("HTTP");

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "service", &service);
	snprintf(file_path, sizeof(file_path), "%s/%s.json", DDNS_SERVICES_DEFAULT, service);

	json_object *json = json_object_from_file(file_path);
	if (!json)
		return 0;

	char *ipv4_url = dmjson_get_value(json, 2, "ipv4", "url");
	if (DM_STRNCMP(ipv4_url, "https", 5) == 0)
		*value = dmstrdup("HTTPS");

	json_object_put(json);

	return 0;
}

static int set_DynamicDNSServer_Protocol(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *SupportedProtocols[] = {"HTTP", "HTTPS", NULL};

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, SupportedProtocols, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			//TODO
			break;
	}
	return 0;
}

/**********************************************************************************************************************************
*                                            OBJ & LEAF DEFINITION
***********************************************************************************************************************************/
/*** Device. ***/
DMOBJ tDeviceObjs[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys*/
{"DynamicDNS", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tDynamicDNSObjs, tDynamicDNSParams, NULL, BBFDM_BOTH, NULL},
{0}
};

/* *** Device.DynamicDNS. *** */
DMOBJ tDynamicDNSObjs[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Client", &DMWRITE, addObjDynamicDNSClient, delObjDynamicDNSClient, NULL, browseDynamicDNSClientInst, NULL, NULL, tDynamicDNSClientObj, tDynamicDNSClientParams, NULL, BBFDM_BOTH, NULL},
{"Server", &DMWRITE, addObjDynamicDNSServer, delObjDynamicDNSServer, NULL, browseDynamicDNSServerInst, NULL, NULL, NULL, tDynamicDNSServerParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tDynamicDNSParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"ClientNumberOfEntries", &DMREAD, DMT_UNINT, get_DynamicDNS_ClientNumberOfEntries, NULL, BBFDM_BOTH},
{"ServerNumberOfEntries", &DMREAD, DMT_UNINT, get_DynamicDNS_ServerNumberOfEntries, NULL, BBFDM_BOTH},
{"SupportedServices", &DMREAD, DMT_STRING, get_DynamicDNS_SupportedServices, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.DynamicDNS.Client.{i}. *** */
DMOBJ tDynamicDNSClientObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Hostname", &DMWRITE, addObjDynamicDNSClientHostname, delObjDynamicDNSClientHostname, NULL, browseDynamicDNSClientHostnameInst, NULL, NULL, NULL, tDynamicDNSClientHostnameParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tDynamicDNSClientParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Enable", &DMWRITE, DMT_BOOL, get_DynamicDNSClient_Enable, set_DynamicDNSClient_Enable, BBFDM_BOTH},
{"Status", &DMREAD, DMT_STRING, get_DynamicDNSClient_Status, NULL, BBFDM_BOTH},
{"Alias", &DMWRITE, DMT_STRING, get_DynamicDNSClient_Alias, set_DynamicDNSClient_Alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"LastError", &DMREAD, DMT_STRING, get_DynamicDNSClient_LastError, NULL, BBFDM_BOTH},
{"Server", &DMWRITE, DMT_STRING, get_DynamicDNSClient_Server, set_DynamicDNSClient_Server, BBFDM_BOTH, DM_FLAG_UNIQUE|DM_FLAG_REFERENCE},
{"Interface", &DMWRITE, DMT_STRING, get_DynamicDNSClient_Interface, set_DynamicDNSClient_Interface, BBFDM_BOTH, DM_FLAG_REFERENCE},
{"Username", &DMWRITE, DMT_STRING, get_DynamicDNSClient_Username, set_DynamicDNSClient_Username, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Password", &DMWRITE, DMT_STRING, get_DynamicDNSClient_Password, set_DynamicDNSClient_Password, BBFDM_BOTH, DM_FLAG_SECURE},
{"HostnameNumberOfEntries", &DMREAD, DMT_UNINT, get_DynamicDNSClient_HostnameNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.DynamicDNS.Client.{i}.Hostname.{i}. *** */
DMLEAF tDynamicDNSClientHostnameParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Enable", &DMWRITE, DMT_BOOL, get_DynamicDNSClientHostname_Enable, set_DynamicDNSClientHostname_Enable, BBFDM_BOTH},
{"Status", &DMREAD, DMT_STRING, get_DynamicDNSClientHostname_Status, NULL, BBFDM_BOTH},
{"Name", &DMWRITE, DMT_STRING, get_DynamicDNSClientHostname_Name, set_DynamicDNSClientHostname_Name, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"LastUpdate", &DMREAD, DMT_TIME, get_DynamicDNSClientHostname_LastUpdate, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.DynamicDNS.Server.{i}. *** */
DMLEAF tDynamicDNSServerParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Enable", &DMWRITE, DMT_BOOL, get_DynamicDNSServer_Enable, set_DynamicDNSServer_Enable, BBFDM_BOTH},
{"Name", &DMWRITE, DMT_STRING, get_DynamicDNSServer_Name, set_DynamicDNSServer_Name, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Alias", &DMWRITE, DMT_STRING, get_DynamicDNSServer_Alias, set_DynamicDNSServer_Alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"ServiceName", &DMWRITE, DMT_STRING, get_DynamicDNSServer_ServiceName, set_DynamicDNSServer_ServiceName, BBFDM_BOTH, DM_FLAG_UNIQUE|DM_FLAG_LINKER},
//{"ServerAddress", &DMWRITE, DMT_STRING, get_DynamicDNSServer_ServerAddress, set_DynamicDNSServer_ServerAddress, BBFDM_BOTH},
//{"ServerPort", &DMWRITE, DMT_UNINT, get_DynamicDNSServer_ServerPort, set_DynamicDNSServer_ServerPort, BBFDM_BOTH},
{"SupportedProtocols", &DMREAD, DMT_STRING, get_DynamicDNSServer_SupportedProtocols, NULL, BBFDM_BOTH},
{"Protocol", &DMWRITE, DMT_STRING, get_DynamicDNSServer_Protocol, set_DynamicDNSServer_Protocol, BBFDM_BOTH},
//{"CheckInterval", &DMWRITE, DMT_UNINT, get_DynamicDNSServer_CheckInterval, set_DynamicDNSServer_CheckInterval, BBFDM_BOTH},
//{"RetryInterval", &DMWRITE, DMT_UNINT, get_DynamicDNSServer_RetryInterval, set_DynamicDNSServer_RetryInterval, BBFDM_BOTH},
//{"MaxRetries", &DMWRITE, DMT_UNINT, get_DynamicDNSServer_MaxRetries, set_DynamicDNSServer_MaxRetries, BBFDM_BOTH},
{0}
};
