/*
 * utils.c - common utility functions
 *
 * Copyright (C) 2022-2024, IOPSYS Software Solutions AB.
 *
 * Author: Amin Ben Romdhane <amin.benromdhane@iopsys.eu>
 *
 * See LICENSE file for license related information.
 *
 */

#include <curl/curl.h>
#include <zlib.h>
#include <libubus.h>
#include <libubox/blob.h>
#include <libubox/blobmsg.h>
#include <libubox/blobmsg_json.h>
#include <libbbfdm-ubus/bbfdm-ubus.h>
#include "utils.h"

#define DM_OBJECT_NAME "bbfdm"

extern struct bbfdm_context bbfdm_ctx;

enum compress_type {
	COMPRESS_TYPE_GZIP,
	COMPRESS_TYPE_COMPRESS,
	COMPRESS_TYPE_DEFLATE
};

static void bulkdata_get_time(char *buffer, size_t len)
{
	if (!buffer || !len)
		return;

	buffer[0] = 0;

	time_t t_time = time(NULL);
	struct tm *t_tm = localtime(&t_time);
	if (t_tm == NULL)
		return;

	strftime(buffer, len, "Date: %a, %d %b %Y %X%z GMT", t_tm);
}

void get_time_stamp(const char *format, time_t coll_time, char **timestamp)
{
	if (strcasecmp(format, "Unix-Epoch") == 0) {
		safe_asprintf(timestamp, "%lld", (long long int)coll_time);
	} else if (strcasecmp(format, "ISO-8601") == 0) {
		char buf[32] = {0};
		struct tm *ts = localtime(&coll_time);
		strftime(buf, sizeof(buf), "%Y-%m-%dT%H:%M:%S%Z", ts);
		safe_asprintf(timestamp, "%s", buf);
	} else
		*timestamp = NULL;
}

unsigned int get_next_period(time_t time_reference, int reporting_interval)
{
	unsigned int next_period;
	time_t now = time(NULL);

	if (now > time_reference)
		next_period = reporting_interval - ((now - time_reference) % reporting_interval);
	else
		next_period = (time_reference - now) % reporting_interval;

	if (next_period == 0)
		next_period = reporting_interval;

	return next_period;
}

static void bulkdata_add_data_to_list(struct list_head *dup_list, const char *name, const char *value, const char *type)
{
	if (dup_list == NULL)
		return;

	param_data_t *link = (param_data_t *)calloc(1, sizeof(param_data_t));
	if (!link)
		return;

	list_add_tail(&link->list, dup_list);
	link->name = strdup(name ? name : "");
	link->data = strdup(value ? value : "");
	link->type = strdup(type ? type : "");
}

void bulkdata_free_data_from_list(struct list_head *dup_list)
{
	param_data_t *link = NULL, *tmp = NULL;

	if (dup_list == NULL || list_empty(dup_list))
		return;

	list_for_each_entry_safe(link, tmp, dup_list, list) {
		list_del(&link->list);
		FREE(link->name);
		FREE(link->data);
		FREE(link->type);
		FREE(link);
	}
}

static int detect_fault(struct blob_attr *msg)
{
	int fault = 0;
	enum {
		FAULT_ID,
		__FAULT_MAX
	};
	struct blob_attr *f_attr[__FAULT_MAX] = {NULL};
	const struct blobmsg_policy fp[__FAULT_MAX] = {
		{ "fault", BLOBMSG_TYPE_INT32 }
	};

	blobmsg_parse(fp, __FAULT_MAX, f_attr, blobmsg_data(msg), blobmsg_len(msg));

	if (f_attr[FAULT_ID])
		fault = blobmsg_get_u32(f_attr[FAULT_ID]);

	return fault;
}

static struct blob_attr *get_parameters(struct blob_attr *msg)
{
	struct blob_attr *params = NULL;
	struct blob_attr *cur;
	int rem;

	blobmsg_for_each_attr(cur, msg, rem) {
		if (blobmsg_type(cur) == BLOBMSG_TYPE_ARRAY) {
			params = cur;
			break;
		}
	}

	return params;
}

static void generate_profile_report(struct profile *profile)
{
	int i;
	bool params_ready = true;

	if (profile == NULL)
		return;

	for (i = 0; i < profile->profile_parameter_number; i++) {
		if (profile->profile_parameter[i].req != NULL) {
			params_ready = false;
			break;
		}
	}

	if (params_ready == false)
		return;

	for (i = 0; i < profile->profile_http_request_uri_parameter_number; i++) {
		if (profile->profile_http_uri_parameter[i].req != NULL) {
			params_ready = false;
			break;
		}
	}

	if (params_ready == false)
		return;

	// All parameters and http parameters value are ready so send report immidiately
	uloop_timeout_cancel(&profile->ctimer);
	bulkdata_generate_report(profile);
}

static void get_value_single_cb(struct ubus_request *req, int type __attribute__((unused)), struct blob_attr *msg)
{
	struct blob_attr *params;
	struct blob_attr *cur;
	size_t rem;
	enum {
		P_PARAM,
		P_VALUE,
		__P_MAX
	};
	const struct blobmsg_policy p[__P_MAX] = {
		{ "path", BLOBMSG_TYPE_STRING },
		{ "data", BLOBMSG_TYPE_STRING }
	};

	if (req == NULL)
		return;

	async_req_t *req_data = (async_req_t *)req->priv;
	if (req_data == NULL)
		return;

	struct profile *profile = (struct profile *)req_data->profile;
	if (profile == NULL) {
		FREE(req_data);
		return;
	}

	profile_http_request_uri_parameter *param = &profile->profile_http_uri_parameter[req_data->param_index];
	param->req = NULL;

	if (msg == NULL)
		goto end;

	if (detect_fault(msg))
		goto end;

	params = get_parameters(msg);
	if (params == NULL)
		goto end;

	blobmsg_for_each_attr(cur, params, rem) {
		struct blob_attr *tb[__P_MAX] = {NULL, NULL};
		char *path, *val;

		blobmsg_parse(p, __P_MAX, tb, blobmsg_data(cur), blobmsg_len(cur));

		if (!tb[P_PARAM] || !tb[P_VALUE])
			continue;

		path = blobmsg_get_string(tb[P_PARAM]);
		val = blobmsg_get_string(tb[P_VALUE]);

		if (strcmp(path, param->reference) == 0) {
			param->value = strdup(val);
			break;
		}
	}

end:
	// Generate and send report if all parameters are ready
	generate_profile_report(profile);

	FREE(req_data);
}

static void get_value_group_cb(struct ubus_request *req, int type __attribute__((unused)), struct blob_attr *msg)
{
	struct blob_attr *params;
	struct blob_attr *cur;
	size_t rem;

	enum {
		P_PARAM,
		P_VALUE,
		P_TYPE,
		__P_MAX
	};
	const struct blobmsg_policy p[__P_MAX] = {
		{ "path", BLOBMSG_TYPE_STRING },
		{ "data", BLOBMSG_TYPE_STRING },
		{ "type", BLOBMSG_TYPE_STRING }
	};

	if (req == NULL)
		return;

	async_req_t *req_data = (async_req_t *)req->priv;
	if (req_data == NULL)
		return;

	struct profile *profile = (struct profile *)req_data->profile;
	if (profile == NULL) {
		FREE(req_data);
		return;
	}

	struct profile_parameter *param = &profile->profile_parameter[req_data->param_index];
	param->req = NULL;

	if (msg == NULL)
		goto end;

	if (detect_fault(msg))
		goto end;

	params = get_parameters(msg);
	if (params == NULL)
		goto end;

	blobmsg_for_each_attr(cur, params, rem) {
		struct blob_attr *tb[__P_MAX] = {NULL, NULL};
		char *path, *val, *ptype;

		blobmsg_parse(p, __P_MAX, tb, blobmsg_data(cur), blobmsg_len(cur));

		if (!tb[P_PARAM] || !tb[P_VALUE] || !tb[P_TYPE])
			continue;

		path = blobmsg_get_string(tb[P_PARAM]);
		val = blobmsg_get_string(tb[P_VALUE]);
		ptype = blobmsg_get_string(tb[P_TYPE]);

		bulkdata_add_data_to_list(param->results, path, val, ptype);
	}

end:
	// Generate and send report if all parameters are ready
	generate_profile_report(profile);

	FREE(req_data);
}

static void complete_callback(struct ubus_request *req, int ret)
{
	FREE(req);
}

static int bbfdm_call_async(struct ubus_context *ctx, const char *method,
		     struct blob_buf *data, ubus_data_handler_t callback,
		     struct profile *profile, int index, int type)
{
	uint32_t id;

	if (profile == NULL || ctx == NULL || method == NULL || data == NULL)
		return -1;

	if (ubus_lookup_id(ctx, DM_OBJECT_NAME, &id))
		return -1;

	async_req_t *req_data = (async_req_t *)malloc(sizeof(async_req_t));
	if (req_data == NULL) {
		ERR("Memory allocation failed for async req data");
		return -1;
	}

	req_data->profile = profile;
	req_data->param_index = index;

	struct ubus_request *req = (struct ubus_request *)malloc(sizeof(struct ubus_request));
	if (req == NULL) {
		ERR("Memory allocation failed for async req");
		FREE(req_data);
		return -1;
	}

	memset(req, 0, sizeof(struct ubus_request));

	// Invoke Ubus to get data from bbfdm
	if (ubus_invoke_async(ctx, id, method, data->head, req)) {
		FREE(req);
		return -1;
	}

	if (callback) {
		req->data_cb = callback;
	}

	req->priv = (void *)req_data;
	req->complete_cb = complete_callback;

	if (type == HTTP_URI_PARAM_REF) {
		profile_http_request_uri_parameter *p = (profile_http_request_uri_parameter *)&profile->profile_http_uri_parameter[index];
		p->req = req;
	} else {
		profile_parameter *p = (profile_parameter *)&profile->profile_parameter[index];
		p->req = req;
	}

	ubus_complete_request_async(ctx, req);

	return 0;
}

int get_value_single(struct ubus_context *ubus_ctx, struct profile *profile, int param_index)
{
	struct blob_buf bb = {0};
	int res = 0;

	if (ubus_ctx == NULL)
		return -1;

	if (profile == NULL)
		return -1;

	profile_http_request_uri_parameter *param = &profile->profile_http_uri_parameter[param_index];

	memset(&bb, 0, sizeof(struct blob_buf));
	blob_buf_init(&bb, 0);

	blobmsg_add_string(&bb, "path", param->reference);
	void *table = blobmsg_open_table(&bb, "optional");
	blobmsg_add_string(&bb, "proto", "cwmp");
	blobmsg_add_string(&bb, "format", "raw");
	blobmsg_close_table(&bb, table);

	// Invoke Ubus to get data from bbfdm
	res = bbfdm_call_async(ubus_ctx, "get", &bb, get_value_single_cb, profile, param_index, HTTP_URI_PARAM_REF);

	blob_buf_free(&bb);
	return res;
}

int get_value_group(struct ubus_context *ubus_ctx, struct profile *profile, int param_index)
{
	struct blob_buf bb = {0};
	int res = 0;

	if (ubus_ctx == NULL)
		return -1;

	if (profile == NULL)
		return -1;

	profile_parameter *param = &profile->profile_parameter[param_index];
	if (param->reference == NULL || strlen(param->reference) == 0)
		return -1;

	memset(&bb, 0, sizeof(struct blob_buf));
	blob_buf_init(&bb, 0);

	blobmsg_add_string(&bb, "path", param->reference);
	void *table = blobmsg_open_table(&bb, "optional");
	blobmsg_add_string(&bb, "proto", "cwmp");
	blobmsg_add_string(&bb, "format", "raw");
	blobmsg_close_table(&bb, table);

	// Invoke Ubus to get data from bbfdm
	res = bbfdm_call_async(ubus_ctx, "get", &bb, get_value_group_cb, profile, param_index, PROFILE_PARAM_REF);

	blob_buf_free(&bb);
	return res;
}

static void create_request_url(struct profile *profile, char *url, size_t len)
{
	int http_uri_number = profile->profile_http_request_uri_parameter_number;
	unsigned pos = 0;

	pos = snprintf(url, len, "%s", profile->http_url);

	for (int i = 0; i < http_uri_number; i++) {
		char *value = NULL;
		struct profile_http_request_uri_parameter *p = &profile->profile_http_uri_parameter[i];

		if (p->name == NULL || strlen(p->name) == 0)
			continue;

		value = p->value ? strdup(p->value) : strdup("");

		if ((len - pos) > (strlen(p->name) + strlen(value))) {
			pos += snprintf(&url[pos], len - pos, "&%s=%s", p->name, value);
		}

		FREE(value);
	}
}

static unsigned char *compress_report(char *input_buf, long input_len, long *content_len, int type)
{
	z_stream zlib_ctx;
	int err;
	int output_len;
	unsigned char *output_buf;

	if (content_len == NULL) {
		return NULL;
	}

	// Exit if not GZIP compression - in this case we just don't compress the report
	if (type != COMPRESS_TYPE_GZIP || input_buf == NULL) {
		*content_len = input_len;
		return (unsigned char *)input_buf;
	}

	// Initialise the zlib context
	memset(&zlib_ctx, 0, sizeof(zlib_ctx));
	zlib_ctx.zalloc = Z_NULL;
	zlib_ctx.zfree = Z_NULL;
	zlib_ctx.opaque = NULL;

	// Exit if unable to start deflate
	#define WINDOW_BITS  (15+16)  // Plus 16 to get a gzip wrapper, as suggested by the zlib documentation
	#define MEM_LEVEL 8           // This is the default value, as suggested by the zlib documentation
	err = deflateInit2(&zlib_ctx, Z_DEFAULT_COMPRESSION, Z_DEFLATED, WINDOW_BITS, MEM_LEVEL, Z_DEFAULT_STRATEGY);
	if (err != Z_OK) {
		WARNING("deflateInit2 returned %d. Falling back to sending uncompressed data", err);
		*content_len = input_len;
		return (unsigned char *)input_buf;
	}

	// Allocate a worst case buffer to hold the compressed data
	output_len = (int)deflateBound(&zlib_ctx, input_len);
	output_buf = malloc(output_len);  // Use malloc because the uncompressed report was generated with malloc() and this needs to be consistent
	if (output_buf == NULL) {
		WARNING("malloc failed. Falling back to sending uncompressed data", err);
		deflateEnd(&zlib_ctx);
		*content_len = input_len;
		return (unsigned char *)input_buf;
	}

	// Initialise the zlib context for this compression
	zlib_ctx.next_in  = (unsigned char *)input_buf;
	zlib_ctx.avail_in = input_len;
	zlib_ctx.next_out = output_buf;
	zlib_ctx.avail_out= output_len;

	// Exit if compression failed
	err = deflate(&zlib_ctx, Z_FINISH);
	if (err != Z_STREAM_END) {
		WARNING("deflate failed (err=%d). Falling back to sending uncompressed data", err);
		deflateEnd(&zlib_ctx);
		free(output_buf);
		*content_len = input_len;
		return (unsigned char *)input_buf;
	}

	// Deallocate all compression state stored in the zlib context
	// NOTE: We ignore errors from this and just log them
	err = deflateEnd(&zlib_ctx);
	if (err != Z_OK) {
		WARNING("deflateEnd failed (err=%d, %s). Ignoring error", err, zlib_ctx.msg);
	}

	INFO("BulkDataReport(uncompressed size=%d, compressed size=%lu)", input_len, zlib_ctx.total_out);
	*content_len = zlib_ctx.total_out;
	return output_buf;
}

int http_send_message(struct profile *profile, char *msg_out)
{
#define HTTP_TIMEOUT 20

	char url[1024] = {0};
	struct curl_slist *header_list = NULL;
	CURLcode res;
	long http_code = 0;
	char errbuf[CURL_ERROR_SIZE];

	CURL *curl = curl_easy_init();
	if (!curl)
		return -1;

	create_request_url(profile, url, sizeof(url));
	INFO("ACS url: %s", url);

	header_list = curl_slist_append(header_list, "User-Agent: iopsys-bulkdata");
	if (!header_list)
		return -1;

	if (profile->http_use_date_header) {
		char local_time[64] = {0};

		bulkdata_get_time(local_time, sizeof(local_time));

		if (strlen(local_time) != 0) {
			header_list = curl_slist_append(header_list, local_time);
			if (!header_list) return -1;
		}
	}

	if (strcasecmp(profile->encoding_type, "JSON") == 0) {
		header_list = curl_slist_append(header_list, "Content-Type: application/json; charset=\"utf-8\"");
		if (!header_list) return -1;

		if (strcasecmp(profile->json_encoding_report_format, "ObjectHierarchy") == 0) {
			header_list = curl_slist_append(header_list, "BBF-Report-Format: ObjectHierarchy");
			if (!header_list) return -1;
		} else if (strcasecmp(profile->json_encoding_report_format, "NameValuePair") == 0) {
			header_list = curl_slist_append(header_list, "BBF-Report-Format: NameValuePair");
			if (!header_list) return -1;
		}
	} else if (strcasecmp(profile->encoding_type, "CSV") == 0) {
		header_list = curl_slist_append(header_list, "Content-Type: text/csv; charset=\"utf-8\"");
		if (!header_list) return -1;

		if (strcasecmp(profile->csv_encoding_report_format, "ParameterPerRow") == 0) {
			header_list = curl_slist_append(header_list, "BBF-Report-Format: ParameterPerRow");
			if (!header_list) return -1;
		} else if (strcasecmp(profile->csv_encoding_report_format, "ParameterPerColumn") == 0) {
			header_list = curl_slist_append(header_list, "BBF-Report-Format: ParameterPerColumn");
			if (!header_list) return -1;
		}
	}

	curl_easy_setopt(curl, CURLOPT_URL, url);
	curl_easy_setopt(curl, CURLOPT_USERNAME, profile->http_username);
	curl_easy_setopt(curl, CURLOPT_PASSWORD, profile->http_password);
	curl_easy_setopt(curl, CURLOPT_HTTPAUTH, CURLAUTH_BASIC|CURLAUTH_DIGEST);
	curl_easy_setopt(curl, CURLOPT_TIMEOUT, HTTP_TIMEOUT);
	curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT, HTTP_TIMEOUT);
	curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1L);
	curl_easy_setopt(curl, CURLOPT_POSTREDIR, CURL_REDIR_POST_ALL);
	curl_easy_setopt(curl, CURLOPT_NOBODY, 0);

	unsigned char *content = NULL;
	long content_len = 0;
	long msg_out_len = msg_out ? strlen(msg_out) : 0;

	if (strcasecmp(profile->http_compression, "GZIP") == 0) {
		curl_easy_setopt(curl, CURLOPT_ACCEPT_ENCODING, "gzip");
		header_list = curl_slist_append(header_list, "Content-Encoding: gzip");
		content = compress_report(msg_out, msg_out_len, &content_len, COMPRESS_TYPE_GZIP);
	} else if (strcasecmp(profile->http_compression, "Compress") == 0) {
		curl_easy_setopt(curl, CURLOPT_ACCEPT_ENCODING, "compress");
		header_list = curl_slist_append(header_list, "Content-Encoding: compress");
		content = compress_report(msg_out, msg_out_len, &content_len, COMPRESS_TYPE_COMPRESS);
	} else if (strcasecmp(profile->http_compression, "Deflate") == 0) {
		curl_easy_setopt(curl, CURLOPT_ACCEPT_ENCODING, "deflate");
		header_list = curl_slist_append(header_list, "Content-Encoding: deflate");
		content = compress_report(msg_out, msg_out_len, &content_len, COMPRESS_TYPE_DEFLATE);
	} else {
		/* no compression */
		content = (unsigned char *)msg_out;
		content_len = msg_out ? msg_out_len : 0;
	}

	curl_easy_setopt(curl, CURLOPT_HTTPHEADER, header_list);
	if (strcasecmp(profile->http_method, "PUT") == 0)
		curl_easy_setopt(curl, CURLOPT_CUSTOMREQUEST, "PUT");

	curl_easy_setopt(curl, CURLOPT_POSTFIELDS, content);
	curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE, content_len);

	curl_easy_setopt(curl, CURLOPT_ERRORBUFFER, errbuf);

	INFO("Send the report of profile name '%s' to Bulkdata Collector", profile->profile_name);
	res = curl_easy_perform(curl);

	if (res != CURLE_OK) {
		size_t len = strlen(errbuf);
		if (len) {
			if (errbuf[len - 1] == '\n') errbuf[len - 1] = '\0';
			WARNING("libcurl: (%d) %s", res, errbuf);
		} else {
			ERR("libcurl: (%d) %s", res, curl_easy_strerror(res));
		}
	}

	if (content != (unsigned char *)msg_out)
		FREE(content);

	curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &http_code);
	if (http_code == 200)
		INFO("Receive HTTP 200 OK from Bulkdata Collector");
	else if (http_code == 401)
		INFO("Receive HTTP 401 Unauthorized from Bulkdata Collector");
	else if (http_code == 204)
		INFO("Receive HTTP 204 No Content from Bulkdata Collector");
	else
		INFO("Receive HTTP %ld from Bulkdata Collector", http_code);

	if (http_code != 200 && http_code != 204)
		goto error;

	if (res)
		goto error;

	curl_easy_cleanup(curl);
	if (header_list) {
		curl_slist_free_all(header_list);
		header_list = NULL;
	}

	return http_code;

error:
	curl_easy_cleanup(curl);
	if (header_list) {
		curl_slist_free_all(header_list);
		header_list = NULL;
	}

	return -1;
}

char *get_bulkdata_profile_parameter_name(const char *paramref, const char *paramname, const char *param)
{
	char buf[512] = {0};
	unsigned pos = 0;

	if (paramname == NULL || strlen(paramname) == 0)
		return strdup(param);
	
	pos = snprintf(buf, sizeof(buf), "%s", paramname);
	
	if (!strchr(paramref, '*')) {
		bool is_same = (paramref[strlen(paramref) - 1] == param[strlen(param) - 1]);
		snprintf(&buf[pos], sizeof(buf) - pos, "%s%s", !is_same ? "." : "", !is_same ? strstr(param, paramref) + strlen(paramref) : "");
	} else {
		unsigned int index = 0;
		char *pch = NULL, *spch = NULL;
		char *idx1 = NULL;
		char paramref_buf[256] = {0};
		char tmp_buf[64] = {0};
		bool first_occur = true;
		
		snprintf(paramref_buf, sizeof(paramref_buf), "%s", paramref);
	
		for (pch = strtok_r(paramref_buf, "*", &spch); pch != NULL; pch = strtok_r(NULL, "*", &spch)) {
		
			if (first_occur) {
				first_occur = false;
				STRNCPY(tmp_buf, pch, sizeof(tmp_buf));
				continue;
			}

			idx1 = strstr(param, tmp_buf);
			sscanf(idx1 + strlen(tmp_buf), "%u.", &index);

			pos += snprintf(&buf[pos], sizeof(buf) - pos, ".%u", index);
			STRNCPY(tmp_buf, pch, sizeof(tmp_buf));
		}

		bool is_same = (tmp_buf[strlen(tmp_buf) - 1] == param[strlen(param) - 1]);
		snprintf(&buf[pos], sizeof(buf) - pos, "%s%s", !is_same ? "." : "", !is_same ? strstr(param, tmp_buf) + strlen(tmp_buf) : "");
	}
		
	return strdup(buf);
}

void append_string_to_string(const char *strappend, char **target)
{
	char *tmp = NULL;

	if (STRLEN(strappend) == 0)
		return;

	if (STRLEN(*target) == 0) {
		*target = strdup(strappend);
		return;
	} else {
		tmp = strdup(*target);
		FREE(*target);
	}
	safe_asprintf(target, "%s%s", tmp, strappend);
	FREE(tmp);
}

void bulkdata_add_failed_report(struct profile *profile, const char *freport)
{
	if (profile->failed_reports == NULL)
		return;

	if (profile->nbre_failed_reports < profile->nbre_of_retained_failed_reports || profile->nbre_of_retained_failed_reports < 0) {
		profile->nbre_failed_reports++;
	} else {
		struct failed_reports *retreport = NULL, *rtmp = NULL;

		list_for_each_entry_safe(retreport, rtmp, profile->failed_reports, list) {
			//bulkdata_delete_failed_report(retreport);
			list_del(&retreport->list);
			FREE(retreport->freport);
			FREE(retreport);
			break;
		}
	}

	struct failed_reports *report = calloc(1, sizeof(struct failed_reports));
	list_add_tail(&report->list, profile->failed_reports);
	report->freport = strdup(freport);
}

struct failed_reports* empty_failed_reports_list(struct profile *profile)
{
	struct failed_reports *report = NULL, *rtmp = NULL;

	if (profile->failed_reports == NULL || list_empty(profile->failed_reports))
		return NULL;

	list_for_each_entry_safe(report, rtmp, profile->failed_reports, list) {
		list_del(&report->list);
		FREE(report->freport);
		FREE(report);
	}
	return NULL;
}

void add_failed_reports_to_report_csv(struct profile *profile, char **report, int isnext)
{
	struct failed_reports *retreport = NULL;
	bool j = false;

	if (profile->failed_reports == NULL || list_empty(profile->failed_reports))
		return;

	list_for_each_entry(retreport, profile->failed_reports, list) {
		if (!j && isnext) {
			j = true;
			continue;
		}
		append_string_to_string(retreport->freport, report);
	}
}

void safe_asprintf(char **strp, const char *fmt, ...)
{
	int ret;
	va_list argp;

	va_start(argp, fmt);
	ret = vasprintf(strp, fmt, argp);
	va_end(argp);

	if (ret == -1) {
		ERR("asprintf failed");
		abort();
	}
}

static void ubus_get_single_parameter_callback(struct ubus_request *req, int type __attribute__((unused)), struct blob_attr *msg)
{
	struct blob_attr *cur = NULL;
	int rem = 0;

	if (msg == NULL || req == NULL)
		return;

	char **result = (char **)req->priv;
	struct blob_attr *parameters = get_parameters(msg);
	if (parameters == NULL)
		return;

	blobmsg_for_each_attr(cur, parameters, rem) {
		struct blob_attr *tb[4] = {0};
		const struct blobmsg_policy p[4] = {
				{ "path", BLOBMSG_TYPE_STRING },
				{ "data", BLOBMSG_TYPE_STRING },
				{ "type", BLOBMSG_TYPE_STRING },
				{ "fault", BLOBMSG_TYPE_INT32 }
		};

		blobmsg_parse(p, 4, tb, blobmsg_data(cur), blobmsg_len(cur));

		if (tb[3]) {
			return;
		}

		*result = strdup(tb[1] ? blobmsg_get_string(tb[1]) : "");
		break;
	}
}

bool bulkdata_get_value(struct ubus_context *ctx, const char *parameter_name, void *value)
{
	struct blob_buf b = {0};
	uint32_t id;
	bool ret = false;

	if (parameter_name == NULL)
		return ret;

	int len = strlen(parameter_name);
	if (len == 0 || parameter_name[len - 1] == '.')
		return ret;

	if (ctx == NULL)
		return ret;

	memset(&b, 0, sizeof(struct blob_buf));
	blob_buf_init(&b, 0);

	blobmsg_add_string(&b, "path", parameter_name);
	void *table = blobmsg_open_table(&b, "optional");
	blobmsg_add_string(&b, "proto", "cwmp");
	blobmsg_add_string(&b, "format", "raw");
	blobmsg_close_table(&b, table);

	if (!ubus_lookup_id(ctx, DM_OBJECT_NAME, &id)) {
		int rc = ubus_invoke(ctx, id, "get", b.head, ubus_get_single_parameter_callback, value, 5000);
		if (rc >= 0) {
			ret = true;
		}
	}

	blob_buf_free(&b);

	return ret;
}

void get_device_id_config(struct ubus_context *ctx, struct device_id *device_id)
{
	char *value = NULL;

	bulkdata_get_value(ctx, "Device.DeviceInfo.ManufacturerOUI", &value);
	device_id->manufacturer_oui = (value) ? strdup(value) : strdup("");
	FREE(value);

	bulkdata_get_value(ctx, "Device.DeviceInfo.ProductClass", &value);
	device_id->product_class = (value) ? strdup(value) : strdup("");
	FREE(value);

	bulkdata_get_value(ctx, "Device.DeviceInfo.SerialNumber", &value);
	device_id->serial_number = (value) ? strdup(value) : strdup("");
	FREE(value);
}

void free_device_id_config(struct device_id *device_id)
{
	FREE(device_id->manufacturer_oui);
	FREE(device_id->product_class);
	FREE(device_id->serial_number);
}

void replace_spaces_with_urichar(struct device_id *device_id)
{
	int space_cnt = 0, len = 0, i;

	len = strlen(device_id->product_class);
	if (len == 0)
		return;

	for (i = 0; i < len; i++) {
		if (device_id->product_class[i] == ' ') {
			space_cnt++;
		}
	}

	if (space_cnt == 0)
		return;

	int new_len = len + (space_cnt * 2) + 1;
	char *value = (char *)calloc(new_len, sizeof(char));
	if (value == NULL) {
		return;
	}

	char *tmp = value;
	for (i = 0; i < len; i++) {
		if (device_id->product_class[i] == ' ') {
			*tmp++ = '%';
			*tmp++ = '2';
			*tmp++ = '0';
		} else {
			*tmp++ = device_id->product_class[i];
		}
	}

	*tmp = '\0';
	FREE(device_id->product_class);

	device_id->product_class = value;
}

void cancel_pending_async_requests(struct profile *profile, bool log_failed)
{
	int i;

	if (profile == NULL)
		return;

	for (i = 0; i < profile->profile_parameter_number; i++) {
		if (profile->profile_parameter[i].req) {
			if (log_failed)
				INFO("Timeout occurred in get of %s", profile->profile_parameter[i].reference);

			ubus_abort_request(bbfdm_ctx.ubus_ctx, profile->profile_parameter[i].req);
			FREE(profile->profile_parameter[i].req->priv);
			FREE(profile->profile_parameter[i].req);
		}
	}

	for (i = 0; i < profile->profile_http_request_uri_parameter_number; i++) {
		if (profile->profile_http_uri_parameter[i].req) {
			if (log_failed)
				INFO("Timeout occurred in get of %s", profile->profile_parameter[i].reference);

			ubus_abort_request(bbfdm_ctx.ubus_ctx, profile->profile_http_uri_parameter[i].req);
			FREE(profile->profile_http_uri_parameter[i].req->priv);
			FREE(profile->profile_http_uri_parameter[i].req);
		}
	}
}
