/*
 * Copyright (C) 2023 iopsys Software Solutions AB
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 2.1
 * as published by the Free Software Foundation
 *
 *	  Author: Suvendhu Hansa <suvendhu.hansa@iopsys.eu>
 *
 */

#include <libbbfdm-api/dmcommon.h>

/*************************************************************
* ENTRY METHOD
*************************************************************/
/*#Device.BulkData.Profile.{i}.!UCI:bulkdata/profile/dmmap_bulkdata*/
static int browseBulkDataProfileInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct dm_data *p = NULL;
	char *inst = NULL;
	LIST_HEAD(dup_list);

	synchronize_specific_config_sections_with_dmmap("bulkdata", "profile", "dmmap_bulkdata", &dup_list);
	list_for_each_entry(p, &dup_list, list) {

		inst = handle_instance(dmctx, parent_node, p->dmmap_section, "profile_instance", "profile_alias");

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)p, inst) == DM_STOP)
			break;
	}
	free_dmmap_config_dup_list(&dup_list);
	return 0;
}

/*#Device.BulkData.Profile.{i}.Parameter.{i}.!UCI:bulkdata/profile_parameter/dmmap_bulkdata*/
static int browseBulkDataProfileParameterInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct uci_section *profile_s = ((struct dm_data *)prev_data)->config_section;
	struct dm_data *p = NULL;
	char *inst = NULL;
	LIST_HEAD(dup_list);

	synchronize_specific_config_sections_with_dmmap_eq("bulkdata", "profile_parameter", "dmmap_bulkdata", "dm_parent", section_name(profile_s), &dup_list);
	list_for_each_entry(p, &dup_list, list) {

		inst = handle_instance(dmctx, parent_node, p->dmmap_section, "parameter_instance", "parameter_alias");

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)p, inst) == DM_STOP)
			break;
	}
	free_dmmap_config_dup_list(&dup_list);
	return 0;
}

/*#Device.BulkData.Profile.{i}.HTTP.RequestURIParameter.{i}.!UCI:bulkdata/profile_http_request_uri_parameter/dmmap_bulkdata*/
static int browseBulkDataProfileHTTPRequestURIParameterInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct uci_section *profile_s = ((struct dm_data *)prev_data)->config_section;
	struct dm_data *p = NULL;
	char *inst = NULL;
	LIST_HEAD(dup_list);

	synchronize_specific_config_sections_with_dmmap_eq("bulkdata", "profile_http_request_uri_parameter", "dmmap_bulkdata", "dm_parent", section_name(profile_s), &dup_list);
	list_for_each_entry(p, &dup_list, list) {

		inst = handle_instance(dmctx, parent_node, p->dmmap_section, "requesturiparameter_instance", "requesturiparameter_alias");

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)p, inst) == DM_STOP)
			break;
	}
	free_dmmap_config_dup_list(&dup_list);
	return 0;
}

/*************************************************************
* ADD & DEL OBJ
*************************************************************/
static int addObjBulkDataProfile(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct uci_section *profile_s = NULL, *dmmap_s = NULL;
	char profile_name[32] = {0};

	snprintf(profile_name, sizeof(profile_name), "profile_%s", *instance);

	dmuci_add_section("bulkdata", "profile", &profile_s);
	dmuci_rename_section_by_section(profile_s, profile_name);

	dmuci_add_section_bbfdm("dmmap_bulkdata", "profile", &dmmap_s);
	dmuci_set_value_by_section(dmmap_s, "section_name", section_name(profile_s));
	dmuci_set_value_by_section(dmmap_s, "profile_instance", *instance);
	return 0;
}

static int delObjBulkDataProfile(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	struct uci_section *s = NULL, *dmmap_section = NULL, *stmp = NULL;
	struct uci_section *profile_s = ((struct dm_data *)data)->config_section;

	// Profile Parameter section
	uci_foreach_option_eq_safe("bulkdata", "profile_parameter", "dm_parent", section_name(profile_s), stmp, s) {
		// Dmmap Profile Parameter section
		get_dmmap_section_of_config_section("dmmap_bulkdata", "profile_parameter", section_name(s), &dmmap_section);
		dmuci_delete_by_section(dmmap_section, NULL, NULL);		
		dmuci_delete_by_section(s, NULL, NULL);
	}

	// Profile HTTP Request URI Parameter section
	uci_foreach_option_eq_safe("bulkdata", "profile_http_request_uri_parameter", "dm_parent", section_name(profile_s), stmp, s) {
		// dmmap Profile HTTP Request URI Parameter section
		get_dmmap_section_of_config_section("dmmap_bulkdata", "profile_http_request_uri_parameter", section_name(s), &dmmap_section);
		dmuci_delete_by_section(dmmap_section, NULL, NULL);

		dmuci_delete_by_section(s, NULL, NULL);
	}

	// Profile section
	dmuci_delete_by_section(((struct dm_data *)data)->config_section, NULL, NULL);

	// Dmmap Profile section
	dmuci_delete_by_section(((struct dm_data *)data)->dmmap_section, NULL, NULL);
	return 0;
}

static int addObjBulkDataProfileParameter(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct uci_section *profile_s = ((struct dm_data *)data)->config_section;
	struct uci_section *profile_param_s = NULL, *dmmap_s = NULL;
	char profile_param_name[32] = {0};

	snprintf(profile_param_name, sizeof(profile_param_name), "%s_param_%s", section_name(profile_s), *instance);

	dmuci_add_section("bulkdata", "profile_parameter", &profile_param_s);
	dmuci_rename_section_by_section(profile_param_s, profile_param_name);
	dmuci_set_value_by_section(profile_param_s, "dm_parent", section_name(profile_s));

	dmuci_add_section_bbfdm("dmmap_bulkdata", "profile_parameter", &dmmap_s);
	dmuci_set_value_by_section(dmmap_s, "section_name", section_name(profile_param_s));
	dmuci_set_value_by_section(dmmap_s, "dm_parent", section_name(profile_s));
	dmuci_set_value_by_section(dmmap_s, "parameter_instance", *instance);
	return 0;
}

static int delObjBulkDataProfileParameter(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	dmuci_delete_by_section(((struct dm_data *)data)->config_section, NULL, NULL);
	dmuci_delete_by_section(((struct dm_data *)data)->dmmap_section, NULL, NULL);
	return 0;
}

static int addObjBulkDataProfileHTTPRequestURIParameter(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct uci_section *profile_s = ((struct dm_data *)data)->config_section;
	struct uci_section *profile_uri_param_s = NULL, *dmmap_s = NULL;
	char profile_uri_param_name[32] = {0};

	snprintf(profile_uri_param_name, sizeof(profile_uri_param_name), "%s_uri_param_%s", section_name(profile_s), *instance);

	dmuci_add_section("bulkdata", "profile_http_request_uri_parameter", &profile_uri_param_s);
	dmuci_rename_section_by_section(profile_uri_param_s, profile_uri_param_name);
	dmuci_set_value_by_section(profile_uri_param_s, "dm_parent", section_name(profile_s));

	dmuci_add_section_bbfdm("dmmap_bulkdata", "profile_http_request_uri_parameter", &dmmap_s);
	dmuci_set_value_by_section(dmmap_s, "section_name", section_name(profile_uri_param_s));
	dmuci_set_value_by_section(dmmap_s, "dm_parent", section_name(profile_s));
	dmuci_set_value_by_section(dmmap_s, "requesturiparameter_instance", *instance);
	return 0;
}

static int delObjBulkDataProfileHTTPRequestURIParameter(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	dmuci_delete_by_section(((struct dm_data *)data)->config_section, NULL, NULL);
	dmuci_delete_by_section(((struct dm_data *)data)->dmmap_section, NULL, NULL);
	return 0;
}

/*************************************************************
* GET & SET PARAM
*************************************************************/
/*#Device.BulkData.Enable!UCI:bulkdata/bulkdata,bulkdata/enable*/
static int get_BulkData_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_option_value_fallback_def("bulkdata", "bulkdata", "enable", "1");
	return 0;
}

static int set_BulkData_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value("bulkdata", "bulkdata", "enable", b ? "1" : "0");
			break;
	}
	return 0;
}

/*#Device.BulkData.Status!UCI:bulkdata/bulkdata,bulkdata/enable*/
static int get_BulkData_Status(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	bool b;

	char *res = dmuci_get_option_value_fallback_def("bulkdata", "bulkdata", "enable", "1");
	string_to_bool(res, &b);

	*value = b ? dmstrdup("Enabled") : dmstrdup("Disabled");

	return 0;
}

static int get_BulkData_MinReportingInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("0");
	return 0;
}

static int get_BulkData_Protocols(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("HTTP");
	return 0;
}

static int get_BulkData_EncodingTypes(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("JSON,CSV");
	return 0;
}

static int get_BulkData_ParameterWildCardSupported(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("1");
	return 0;
}

static int get_BulkData_MaxNumberOfProfiles(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("-1");
	return 0;
}

static int get_BulkData_MaxNumberOfParameterReferences(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("-1");
	return 0;
}

/*#Device.BulkData.ProfileNumberOfEntries!UCI:bulkdata/profile/*/
static int get_BulkData_ProfileNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseBulkDataProfileInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

/*#Device.BulkData.Profile.{i}.Enable!UCI:bulkdata/profile,@i-1/enable*/
static int get_BulkDataProfile_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "enable", "0");
	return 0;
}

static int set_BulkDataProfile_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "enable", b ? "1" : "0");
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.Alias!UCI:dmmap_bulkdata/profile,@i-1/profile_alias*/
static int get_BulkDataProfile_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return bbf_get_alias(ctx, ((struct dm_data *)data)->dmmap_section, "profile_alias", instance, value);;
}

static int set_BulkDataProfile_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	return bbf_set_alias(ctx, ((struct dm_data *)data)->dmmap_section, "profile_alias", instance, value);
}

/*#Device.BulkData.Profile.{i}.Name!UCI:bulkdata/profile,@i-1/name*/
static int get_BulkDataProfile_Name(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "name", value);
	return 0;
}

static int set_BulkDataProfile_Name(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 255, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "name", value);
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.NumberOfRetainedFailedReports!UCI:bulkdata/profile,@i-1/nbre_of_retained_failed_reports*/
static int get_BulkDataProfile_NumberOfRetainedFailedReports(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "nbre_of_retained_failed_reports", value);
	return 0;
}

static int set_BulkDataProfile_NumberOfRetainedFailedReports(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_int(ctx, value, RANGE_ARGS{{"-1",NULL}}, 1))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "nbre_of_retained_failed_reports", value);
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.Protocol!UCI:bulkdata/profile,@i-1/protocol*/
static int get_BulkDataProfile_Protocol(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "protocol", "HTTP");
	return 0;
}

static int set_BulkDataProfile_Protocol(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *allowed_values[] = {"HTTP", NULL};
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, allowed_values, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "protocol", value);
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.EncodingType!UCI:bulkdata/profile,@i-1/encoding_type*/
static int get_BulkDataProfile_EncodingType(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "encoding_type", value);
	return 0;
}

static int set_BulkDataProfile_EncodingType(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *allowed_values[] = {"CSV", "JSON", NULL};
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, allowed_values, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "encoding_type", value);
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.ReportingInterval!UCI:bulkdata/profile,@i-1/reporting_interval*/
static int get_BulkDataProfile_ReportingInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "reporting_interval", "86400");
	return 0;
}

static int set_BulkDataProfile_ReportingInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"1",NULL}}, 1))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "reporting_interval", value);
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.TimeReference!UCI:bulkdata/profile,@i-1/time_reference*/
static int get_BulkDataProfile_TimeReference(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	time_t time_value;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "time_reference", value);
	if ((*value)[0] != '0' && (*value)[0] != '\0') {
		time_value = (time_t)strtol(*value, NULL, 10);
		char s_now[sizeof "AAAA-MM-JJTHH:MM:SSZ"];
		strftime(s_now, sizeof s_now, "%Y-%m-%dT%H:%M:%SZ", localtime(&time_value));
		*value = dmstrdup(s_now);
	} else {
		*value = dmstrdup("0001-01-01T00:00:00Z");
	}
	return 0;
}

static int set_BulkDataProfile_TimeReference(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct tm tm;
	char buf[16];

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_dateTime(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			if (!(strptime(value, "%Y-%m-%dT%H:%M:%S", &tm)))
				break;
			snprintf(buf, sizeof(buf), "%lld", (long long int)mktime(&tm));
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "time_reference", buf);
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.ParameterNumberOfEntries!UCI:bulkdata/profile_parameter,false/false*/
static int get_BulkDataProfile_ParameterNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseBulkDataProfileParameterInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

/*#Device.BulkData.Profile.{i}.Parameter.{i}.Name!UCI:bulkdata/profile_parameter,@i-1/name*/
static int get_BulkDataProfileParameter_Name(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "name", value);
	return 0;
}

static int set_BulkDataProfileParameter_Name(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 64, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "name", value);
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.Parameter.{i}.Reference!UCI:bulkdata/profile_parameter,@i-1/reference*/
static int get_BulkDataProfileParameter_Reference(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "reference", value);
	return 0;
}

static int set_BulkDataProfileParameter_Reference(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 256, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "reference", value);
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.CSVEncoding.FieldSeparator!UCI:bulkdata/profile,@i-1/csv_encoding_field_separator*/
static int get_BulkDataProfileCSVEncoding_FieldSeparator(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "csv_encoding_field_separator", value);
	return 0;
}

static int set_BulkDataProfileCSVEncoding_FieldSeparator(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "csv_encoding_field_separator", value);
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.CSVEncoding.RowSeparator!UCI:bulkdata/profile,@i-1/csv_encoding_row_separator*/
static int get_BulkDataProfileCSVEncoding_RowSeparator(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "csv_encoding_row_separator", value);
	return 0;
}

static int set_BulkDataProfileCSVEncoding_RowSeparator(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "csv_encoding_row_separator", value);
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.CSVEncoding.EscapeCharacter!UCI:bulkdata/profile,@i-1/csv_encoding_escape_character*/
static int get_BulkDataProfileCSVEncoding_EscapeCharacter(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "csv_encoding_escape_character", value);
	return 0;
}

static int set_BulkDataProfileCSVEncoding_EscapeCharacter(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "csv_encoding_escape_character", value);
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.CSVEncoding.ReportFormat!UCI:bulkdata/profile,@i-1/csv_encoding_report_format*/
static int get_BulkDataProfileCSVEncoding_ReportFormat(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "csv_encoding_report_format", "ParameterPerColumn");
	return 0;
}

static int set_BulkDataProfileCSVEncoding_ReportFormat(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *allowed_values[] = {"ParameterPerRow", "ParameterPerColumn", NULL};

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, allowed_values, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "csv_encoding_report_format", value);
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.CSVEncoding.RowTimestamp!UCI:bulkdata/profile,@i-1/csv_encoding_row_time_stamp*/
static int get_BulkDataProfileCSVEncoding_RowTimestamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "csv_encoding_row_time_stamp", "Unix-Epoch");
	return 0;
}

static int set_BulkDataProfileCSVEncoding_RowTimestamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *allowed_values[] = {"Unix-Epoch", "ISO-8601", "None", NULL};
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, allowed_values, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "csv_encoding_row_time_stamp", value);
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.JSONEncoding.ReportFormat!UCI:bulkdata/profile,@i-1/json_encoding_report_format*/
static int get_BulkDataProfileJSONEncoding_ReportFormat(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "json_encoding_report_format", "ObjectHierarchy");
	return 0;
}

static int set_BulkDataProfileJSONEncoding_ReportFormat(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *allowed_values[] = {"ObjectHierarchy", "NameValuePair", NULL};

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, allowed_values, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "json_encoding_report_format", value);
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.JSONEncoding.ReportTimestamp!UCI:bulkdata/profile,@i-1/json_encoding_report_time_stamp*/
static int get_BulkDataProfileJSONEncoding_ReportTimestamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "json_encoding_report_time_stamp", "Unix-Epoch");
	return 0;
}

static int set_BulkDataProfileJSONEncoding_ReportTimestamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *allowed_values[] = {"Unix-Epoch", "ISO-8601", "None", NULL};
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, allowed_values, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "json_encoding_report_time_stamp", value);
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.HTTP.URL!UCI:bulkdata/profile,@i-1/http_url*/
static int get_BulkDataProfileHTTP_URL(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "http_url", value);
	return 0;
}

static int set_BulkDataProfileHTTP_URL(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 2048, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "http_url", value);
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.HTTP.Username!UCI:bulkdata/profile,@i-1/http_username*/
static int get_BulkDataProfileHTTP_Username(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "http_username", value);
	return 0;
}

static int set_BulkDataProfileHTTP_Username(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 256, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "http_username", value);
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.HTTP.Password!UCI:bulkdata/profile,@i-1/http_password*/
static int get_BulkDataProfileHTTP_Password(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return 0;
}

static int set_BulkDataProfileHTTP_Password(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 256, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "http_password", value);
			break;
	}
	return 0;
}

static int get_BulkDataProfileHTTP_CompressionsSupported(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("GZIP,None");
	return 0;
}

/*#Device.BulkData.Profile.{i}.HTTP.Compression!UCI:bulkdata/profile,@i-1/http_compression*/
static int get_BulkDataProfileHTTP_Compression(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "http_compression", "None");
	return 0;
}

static int set_BulkDataProfileHTTP_Compression(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *allowed_values[] = {"GZIP", "None", NULL};

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, allowed_values, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "http_compression", value);
			break;
	}
	return 0;
}

static int get_BulkDataProfileHTTP_MethodsSupported(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("POST,PUT");
	return 0;
}

/*#Device.BulkData.Profile.{i}.HTTP.Method!UCI:bulkdata/profile,@i-1/http_method*/
static int get_BulkDataProfileHTTP_Method(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "http_method", "POST");
	return 0;
}

static int set_BulkDataProfileHTTP_Method(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *allowed_values[] = {"POST", "PUT", NULL};

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, allowed_values, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "http_method", value);
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.HTTP.UseDateHeader!UCI:bulkdata/profile,@i-1/http_use_date_header*/
static int get_BulkDataProfileHTTP_UseDateHeader(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "http_use_date_header", "1");
	return 0;
}

static int set_BulkDataProfileHTTP_UseDateHeader(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "http_use_date_header", b ? "1" : "0");
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.HTTP.RetryEnable!UCI:bulkdata/profile,@i-1/http_retry_enable*/
static int get_BulkDataProfileHTTP_RetryEnable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "http_retry_enable", "0");
	return 0;
}

static int set_BulkDataProfileHTTP_RetryEnable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "http_retry_enable", b ? "1" : "0");
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.HTTP.RetryMinimumWaitInterval!UCI:bulkdata/profile,@i-1/http_retry_minimum_wait_interval*/
static int get_BulkDataProfileHTTP_RetryMinimumWaitInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "http_retry_minimum_wait_interval", "5");
	return 0;
}

static int set_BulkDataProfileHTTP_RetryMinimumWaitInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"1","65535"}}, 1))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "http_retry_minimum_wait_interval", value);
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.HTTP.RetryIntervalMultiplier!UCI:bulkdata/profile,@i-1/http_retry_interval_multiplier*/
static int get_BulkDataProfileHTTP_RetryIntervalMultiplier(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "http_retry_interval_multiplier", "2000");
	return 0;
}

static int set_BulkDataProfileHTTP_RetryIntervalMultiplier(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"1000","65535"}}, 1))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "http_retry_interval_multiplier", value);
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.HTTP.RequestURIParameterNumberOfEntries!UCI:bulkdata/profile_http_request_uri_parameter,false/false*/
static int get_BulkDataProfileHTTP_RequestURIParameterNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseBulkDataProfileHTTPRequestURIParameterInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

/*#Device.BulkData.Profile.{i}.HTTP.PersistAcrossReboot!UCI:bulkdata/profile,@i-1/http_persist_across_reboot*/
static int get_BulkDataProfileHTTP_PersistAcrossReboot(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "http_persist_across_reboot", "0");
	return 0;
}

static int set_BulkDataProfileHTTP_PersistAcrossReboot(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "http_persist_across_reboot", b ? "1" : "0");
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.HTTP.RequestURIParameter.{i}.Name!UCI:bulkdata/profile_http_request_uri_parameter,@i-1/name*/
static int get_BulkDataProfileHTTPRequestURIParameter_Name(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "name", value);
	return 0;
}

static int set_BulkDataProfileHTTPRequestURIParameter_Name(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 64, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "name", value);
			break;
	}
	return 0;
}

/*#Device.BulkData.Profile.{i}.HTTP.RequestURIParameter.{i}.Reference!UCI:bulkdata/profile_http_request_uri_parameter,@i-1/reference*/
static int get_BulkDataProfileHTTPRequestURIParameter_Reference(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "reference", value);
	return 0;
}

static int set_BulkDataProfileHTTPRequestURIParameter_Reference(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 256, NULL, NULL))
				return FAULT_9007;

			int len = DM_STRLEN(value);
			if (len == 0)
				break;

			if (value[len - 1] == '.') {
				bbfdm_set_fault_message(ctx, "This should be a reference path of a parameter");
				return FAULT_9007;
			}
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "reference", value);
			break;
	}
	return 0;
}


/**********************************************************************************************************************************
*                                            OBJ & LEAF DEFINITION
***********************************************************************************************************************************/
/* *** Device.BulkData.Profile.{i}.HTTP.RequestURIParameter.{i}. *** */
DMLEAF tBulkDataProfileHTTPRequestURIParameterParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{"Name", &DMWRITE, DMT_STRING, get_BulkDataProfileHTTPRequestURIParameter_Name, set_BulkDataProfileHTTPRequestURIParameter_Name, BBFDM_CWMP},
{"Reference", &DMWRITE, DMT_STRING, get_BulkDataProfileHTTPRequestURIParameter_Reference, set_BulkDataProfileHTTPRequestURIParameter_Reference, BBFDM_CWMP},
{0}
};

DMLEAF tBulkDataProfileHTTPParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{"URL", &DMWRITE, DMT_STRING, get_BulkDataProfileHTTP_URL, set_BulkDataProfileHTTP_URL, BBFDM_CWMP},
{"Username", &DMWRITE, DMT_STRING, get_BulkDataProfileHTTP_Username, set_BulkDataProfileHTTP_Username, BBFDM_CWMP},
{"Password", &DMWRITE, DMT_STRING, get_BulkDataProfileHTTP_Password, set_BulkDataProfileHTTP_Password, BBFDM_CWMP},
{"CompressionsSupported", &DMREAD, DMT_STRING, get_BulkDataProfileHTTP_CompressionsSupported, NULL, BBFDM_CWMP},
{"Compression", &DMWRITE, DMT_STRING, get_BulkDataProfileHTTP_Compression, set_BulkDataProfileHTTP_Compression, BBFDM_CWMP},
{"MethodsSupported", &DMREAD, DMT_STRING, get_BulkDataProfileHTTP_MethodsSupported, NULL, BBFDM_CWMP},
{"Method", &DMWRITE, DMT_STRING, get_BulkDataProfileHTTP_Method, set_BulkDataProfileHTTP_Method, BBFDM_CWMP},
{"UseDateHeader", &DMWRITE, DMT_BOOL, get_BulkDataProfileHTTP_UseDateHeader, set_BulkDataProfileHTTP_UseDateHeader, BBFDM_CWMP},
{"RetryEnable", &DMWRITE, DMT_BOOL, get_BulkDataProfileHTTP_RetryEnable, set_BulkDataProfileHTTP_RetryEnable, BBFDM_CWMP},
{"RetryMinimumWaitInterval", &DMWRITE, DMT_UNINT, get_BulkDataProfileHTTP_RetryMinimumWaitInterval, set_BulkDataProfileHTTP_RetryMinimumWaitInterval, BBFDM_CWMP},
{"RetryIntervalMultiplier", &DMWRITE, DMT_UNINT, get_BulkDataProfileHTTP_RetryIntervalMultiplier, set_BulkDataProfileHTTP_RetryIntervalMultiplier, BBFDM_CWMP},
{"RequestURIParameterNumberOfEntries", &DMREAD, DMT_UNINT, get_BulkDataProfileHTTP_RequestURIParameterNumberOfEntries, NULL, BBFDM_CWMP},
{"PersistAcrossReboot", &DMWRITE, DMT_BOOL, get_BulkDataProfileHTTP_PersistAcrossReboot, set_BulkDataProfileHTTP_PersistAcrossReboot, BBFDM_CWMP},
{0}
};

/* *** Device.BulkData.Profile.{i}.HTTP. *** */
DMOBJ tBulkDataProfileHTTPObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys*/
{"RequestURIParameter", &DMWRITE, addObjBulkDataProfileHTTPRequestURIParameter, delObjBulkDataProfileHTTPRequestURIParameter, NULL, browseBulkDataProfileHTTPRequestURIParameterInst, NULL, NULL, NULL, tBulkDataProfileHTTPRequestURIParameterParams, NULL, BBFDM_CWMP},
{0}
};

/* *** Device.BulkData.Profile.{i}.JSONEncoding. *** */
DMLEAF tBulkDataProfileJSONEncodingParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{"ReportFormat", &DMWRITE, DMT_STRING, get_BulkDataProfileJSONEncoding_ReportFormat, set_BulkDataProfileJSONEncoding_ReportFormat, BBFDM_CWMP},
{"ReportTimestamp", &DMWRITE, DMT_STRING, get_BulkDataProfileJSONEncoding_ReportTimestamp, set_BulkDataProfileJSONEncoding_ReportTimestamp, BBFDM_CWMP},
{0}
};

/* *** Device.BulkData.Profile.{i}.CSVEncoding. *** */
DMLEAF tBulkDataProfileCSVEncodingParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{"FieldSeparator", &DMWRITE, DMT_STRING, get_BulkDataProfileCSVEncoding_FieldSeparator, set_BulkDataProfileCSVEncoding_FieldSeparator, BBFDM_CWMP},
{"RowSeparator", &DMWRITE, DMT_STRING, get_BulkDataProfileCSVEncoding_RowSeparator, set_BulkDataProfileCSVEncoding_RowSeparator, BBFDM_CWMP},
{"EscapeCharacter", &DMWRITE, DMT_STRING, get_BulkDataProfileCSVEncoding_EscapeCharacter, set_BulkDataProfileCSVEncoding_EscapeCharacter, BBFDM_CWMP},
{"ReportFormat", &DMWRITE, DMT_STRING, get_BulkDataProfileCSVEncoding_ReportFormat, set_BulkDataProfileCSVEncoding_ReportFormat, BBFDM_CWMP},
{"RowTimestamp", &DMWRITE, DMT_STRING, get_BulkDataProfileCSVEncoding_RowTimestamp, set_BulkDataProfileCSVEncoding_RowTimestamp, BBFDM_CWMP},
{0}
};

/* *** Device.BulkData.Profile.{i}.Parameter.{i}. *** */
DMLEAF tBulkDataProfileParameterParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{"Name", &DMWRITE, DMT_STRING, get_BulkDataProfileParameter_Name, set_BulkDataProfileParameter_Name, BBFDM_CWMP},
{"Reference", &DMWRITE, DMT_STRING, get_BulkDataProfileParameter_Reference, set_BulkDataProfileParameter_Reference, BBFDM_CWMP},
{0}
};

DMLEAF tBulkDataProfileParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{"Enable", &DMWRITE, DMT_BOOL, get_BulkDataProfile_Enable, set_BulkDataProfile_Enable, BBFDM_CWMP},
{"Alias", &DMWRITE, DMT_STRING, get_BulkDataProfile_Alias, set_BulkDataProfile_Alias, BBFDM_CWMP, DM_FLAG_UNIQUE},
{"Name", &DMWRITE, DMT_STRING, get_BulkDataProfile_Name, set_BulkDataProfile_Name, BBFDM_CWMP},
{"NumberOfRetainedFailedReports", &DMWRITE, DMT_INT, get_BulkDataProfile_NumberOfRetainedFailedReports, set_BulkDataProfile_NumberOfRetainedFailedReports, BBFDM_CWMP},
{"Protocol", &DMWRITE, DMT_STRING, get_BulkDataProfile_Protocol, set_BulkDataProfile_Protocol, BBFDM_CWMP},
{"EncodingType", &DMWRITE, DMT_STRING, get_BulkDataProfile_EncodingType, set_BulkDataProfile_EncodingType, BBFDM_CWMP},
{"ReportingInterval", &DMWRITE, DMT_UNINT, get_BulkDataProfile_ReportingInterval, set_BulkDataProfile_ReportingInterval, BBFDM_CWMP},
{"TimeReference", &DMWRITE, DMT_TIME, get_BulkDataProfile_TimeReference, set_BulkDataProfile_TimeReference, BBFDM_CWMP},
{"ParameterNumberOfEntries", &DMREAD, DMT_UNINT, get_BulkDataProfile_ParameterNumberOfEntries, NULL, BBFDM_CWMP},
{0}
};

/* *** Device.BulkData.Profile.{i}. *** */
DMOBJ tBulkDataProfileObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys*/
{"Parameter", &DMWRITE, addObjBulkDataProfileParameter, delObjBulkDataProfileParameter, NULL, browseBulkDataProfileParameterInst, NULL, NULL, NULL, tBulkDataProfileParameterParams, NULL, BBFDM_CWMP},
{"CSVEncoding", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tBulkDataProfileCSVEncodingParams, NULL, BBFDM_CWMP},
{"JSONEncoding", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tBulkDataProfileJSONEncodingParams, NULL, BBFDM_CWMP},
{"HTTP", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tBulkDataProfileHTTPObj, tBulkDataProfileHTTPParams, NULL, BBFDM_CWMP},
{0}
};

DMOBJ tBulkDataObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys*/
{"Profile", &DMWRITE, addObjBulkDataProfile, delObjBulkDataProfile, NULL, browseBulkDataProfileInst, NULL, NULL, tBulkDataProfileObj, tBulkDataProfileParams, NULL, BBFDM_CWMP, NULL},
{0}
};

DMLEAF tBulkDataParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{"Enable", &DMWRITE, DMT_BOOL, get_BulkData_Enable, set_BulkData_Enable, BBFDM_CWMP},
{"Status", &DMREAD, DMT_STRING, get_BulkData_Status, NULL, BBFDM_CWMP},
{"MinReportingInterval", &DMREAD, DMT_UNINT, get_BulkData_MinReportingInterval, NULL, BBFDM_CWMP},
{"Protocols", &DMREAD, DMT_STRING, get_BulkData_Protocols, NULL, BBFDM_CWMP},
{"EncodingTypes", &DMREAD, DMT_STRING, get_BulkData_EncodingTypes, NULL, BBFDM_CWMP},
{"ParameterWildCardSupported", &DMREAD, DMT_BOOL, get_BulkData_ParameterWildCardSupported, NULL, BBFDM_CWMP},
{"MaxNumberOfProfiles", &DMREAD, DMT_INT, get_BulkData_MaxNumberOfProfiles, NULL, BBFDM_CWMP},
{"MaxNumberOfParameterReferences", &DMREAD, DMT_INT, get_BulkData_MaxNumberOfParameterReferences, NULL, BBFDM_CWMP},
{"ProfileNumberOfEntries", &DMREAD, DMT_UNINT, get_BulkData_ProfileNumberOfEntries, NULL, BBFDM_CWMP},
{0}
};

/* *** Device.BulkData. *** */
DMOBJ tDeviceBulkDataObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys*/
{"BulkData", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tBulkDataObj, tBulkDataParams, NULL, BBFDM_CWMP},
{0}
};

/* ********** DynamicObj ********** */
DM_MAP_OBJ tDynamicObj[] = {
/* parentobj, nextobject, parameter */
{"Device.", tDeviceBulkDataObj, NULL},
{0}
};
