/*
 * config.c - common utility functions
 *
 * Copyright (C) 2022-2024, IOPSYS Software Solutions AB.
 *
 * Author: Amin Ben Romdhane <amin.benromdhane@iopsys.eu>
 * Author: Omar Kallel <omar.kallel@pivasoftware.com>
 *
 * See LICENSE file for license related information.
 *
 */

#include <stdio.h>
#include <strings.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>

#include "config.h"
#include "buci.h"
#include "utils.h"

static void get_bulkdata_enable(struct bulkdata *bulkdata)
{
	buci_init();
	char *enable = buci_get_value("bulkdata", "bulkdata", "enable");

	if (strcmp(enable, "0") == 0 || strcmp(enable, "false") == 0) {
		DEBUG("Bulkdata service disabled");
		bulkdata->enable = false;
	} else {
		DEBUG("Bulkdata service enabled");
		bulkdata->enable = true;
	}
	buci_fini();
}

static void load_profile_config(struct bulkdata *bulkdata, struct uci_section *s, int idx, struct device_id *device_id)
{
	char http_url[128] = {0};
	char *data = NULL;

	bulkdata->profile[idx].profile_name = strdup(s ? (s)->e.name : "profile_1");
	DEBUG("The profile name of profile index %d is : %s", idx, bulkdata->profile[idx].profile_name);

	data = buci_get_value_bysection(s, "nbre_of_retained_failed_reports");
	bulkdata->profile[idx].nbre_of_retained_failed_reports = (STRLEN(data)) ? (int)strtol(data, NULL, 10) : 0;
	DEBUG("The nombre of retained failed reports of profile name '%s' is : %d", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].nbre_of_retained_failed_reports);

	data = buci_get_value_bysection(s, "protocol");
	bulkdata->profile[idx].protocol = strdup((STRLEN(data) && !strcasecmp(data, "http")) ? data : "http");
	DEBUG("The protocol of profile name '%s' is : %s", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].protocol);

	data = buci_get_value_bysection(s, "encoding_type");
	bulkdata->profile[idx].encoding_type = strdup((STRLEN(data)) ? data : "");
	DEBUG("The encoding type of profile name '%s' is : %s", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].encoding_type);

	data = buci_get_value_bysection(s, "reporting_interval");
	bulkdata->profile[idx].reporting_interval = (STRLEN(data)) ? (int)strtol(data, NULL, 10) : 86400;
	DEBUG("The reporting interval of profile name '%s' is : %d", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].reporting_interval);

	data = buci_get_value_bysection(s, "time_reference");
	bulkdata->profile[idx].time_reference = (STRLEN(data)) ? (int)strtol(data, NULL, 10) : 0;
	DEBUG("The time reference of profile name '%s' is : %ld", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].time_reference);

	data = buci_get_value_bysection(s, "csv_encoding_field_separator");
	bulkdata->profile[idx].csv_encoding_field_separator = strdup((STRLEN(data)) ? data : ",");
	DEBUG("The csv encoding field separator of profile name '%s' is : %s", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].csv_encoding_field_separator);

	data = buci_get_value_bysection(s, "csv_encoding_row_separator");
	bulkdata->profile[idx].csv_encoding_row_separator = strdup((STRLEN(data)) ? data : "&#10;");
	DEBUG("The csv encoding row separator of profile name '%s' is : %s", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].csv_encoding_row_separator);

	data = buci_get_value_bysection(s, "csv_encoding_escape_character");
	bulkdata->profile[idx].csv_encoding_escape_character = strdup((STRLEN(data)) ? data : "&quot;");
	DEBUG("The csv encoding escape character of profile name '%s' is : %s", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].csv_encoding_escape_character);

	data = buci_get_value_bysection(s, "csv_encoding_report_format");
	bulkdata->profile[idx].csv_encoding_report_format = strdup((STRLEN(data)) ? data : "ParameterPerColumn");
	DEBUG("The csv encoding report format of profile name '%s' is : %s", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].csv_encoding_report_format);

	data = buci_get_value_bysection(s, "csv_encoding_row_time_stamp");
	bulkdata->profile[idx].csv_encoding_row_time_stamp = strdup((STRLEN(data)) ? data : "Unix-Epoch");
	DEBUG("The csv encoding row time stamp of profile name '%s' is : %s", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].csv_encoding_row_time_stamp);

	data = buci_get_value_bysection(s, "json_encoding_report_format");
	bulkdata->profile[idx].json_encoding_report_format = strdup((STRLEN(data)) ? data : "ObjectHierarchy");
	DEBUG("The json encoding report format of profile name '%s' is : %s", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].json_encoding_report_format);

	data = buci_get_value_bysection(s, "json_encoding_report_time_stamp");
	bulkdata->profile[idx].json_encoding_report_time_stamp = strdup((STRLEN(data)) ? data : "Unix-Epoch");
	DEBUG("The json encoding report time stamp of profile name '%s' is : %s", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].json_encoding_report_time_stamp);

	data = buci_get_value_bysection(s, "http_url");
	snprintf(http_url, sizeof(http_url), "%s?oui=%s&pc=%s&sn=%s", (STRLEN(data)) ? data : "", device_id->manufacturer_oui,
									device_id->product_class, device_id->serial_number);
	bulkdata->profile[idx].http_url = strdup(http_url);
	DEBUG("The HTTP url of profile name '%s' is : %s", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].http_url);

	data = buci_get_value_bysection(s, "http_username");
	bulkdata->profile[idx].http_username = strdup((STRLEN(data)) ? data : "");
	DEBUG("The HTTP username of profile name '%s' is : %s", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].http_username);

	data = buci_get_value_bysection(s, "http_password");
	bulkdata->profile[idx].http_password = strdup((STRLEN(data)) ? data : "");
	DEBUG("The HTTP password of profile name '%s' is : %s", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].http_password);

	data = buci_get_value_bysection(s, "http_compression");
	bulkdata->profile[idx].http_compression = strdup((STRLEN(data)) ? data : "None");
	DEBUG("The HTTP compression of profile name '%s' is : %s", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].http_compression);

	data = buci_get_value_bysection(s, "http_method");
	bulkdata->profile[idx].http_method = strdup((STRLEN(data)) ? data : "POST");
	DEBUG("The HTTP method of profile name '%s' is : %s", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].http_method);

	data = buci_get_value_bysection(s, "http_use_date_header");
	bulkdata->profile[idx].http_use_date_header = (STRLEN(data)) ? (bool)strtol(data, NULL, 10) : 1;
	DEBUG("The HTTP use date header of profile name '%s' is : %d", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].http_use_date_header);

	data = buci_get_value_bysection(s, "http_retry_enable");
	bulkdata->profile[idx].http_retry_enable = (STRLEN(data)) ? (bool)strtol(data, NULL, 10) : 0;
	DEBUG("The HTTP retry enable of profile name '%s' is : %d", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].http_retry_enable);

	data = buci_get_value_bysection(s, "http_retry_minimum_wait_interval");
	bulkdata->profile[idx].http_retry_minimum_wait_interval = (STRLEN(data)) ? (int)strtol(data, NULL, 10) : 5;
	DEBUG("The HTTP retry minimum wait interval of profile name '%s' is : %d", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].http_retry_minimum_wait_interval);

	data = buci_get_value_bysection(s, "http_retry_interval_multiplier");
	bulkdata->profile[idx].http_retry_interval_multiplier = (STRLEN(data)) ? (int)strtol(data, NULL, 10) : 2000;
	DEBUG("The HTTP retry interval multiplier of profile name '%s' is : %d", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].http_retry_interval_multiplier);

	data = buci_get_value_bysection(s, "http_persist_across_reboot");
	bulkdata->profile[idx].http_persist_across_reboot = (STRLEN(data)) ? (bool)strtol(data, NULL, 10) : 0;
	DEBUG("The HTTP persist across reboot of profile name '%s' is : %d", bulkdata->profile[idx].profile_name, bulkdata->profile[idx].http_persist_across_reboot);

	bulkdata->profile[idx].retry_count = 0;
	bulkdata->profile[idx].nbre_failed_reports = 0;
}

static void load_profile_parameter_config(struct bulkdata *bulkdata, struct uci_section *s, int p_idx, int idx)
{
	char *data = NULL;

	bulkdata->profile[p_idx].profile_parameter[idx].results = calloc(1, sizeof(struct list_head));
	if (bulkdata->profile[p_idx].profile_parameter[idx].results != NULL) {
		INIT_LIST_HEAD(bulkdata->profile[p_idx].profile_parameter[idx].results);
	} else {
		ERR("Failed to allocate memory for storing values of profile parameter");
	}

	bulkdata->profile[p_idx].profile_parameter[idx].req = NULL;

	data = buci_get_value_bysection(s, "name");
	bulkdata->profile[p_idx].profile_parameter[idx].name = strdup((STRLEN(data)) ? data : "");
	DEBUG("The parameter name %d of profile name '%s' is : %s", idx+1, bulkdata->profile[p_idx].profile_name, bulkdata->profile[p_idx].profile_parameter[idx].name);

	data = buci_get_value_bysection(s, "reference");
	bulkdata->profile[p_idx].profile_parameter[idx].reference = strdup((STRLEN(data)) ? data : "");;
	DEBUG("The parameter reference %d of profile name '%s' is : %s", idx+1, bulkdata->profile[p_idx].profile_name, bulkdata->profile[p_idx].profile_parameter[idx].reference);
}

static void get_profiles_parameters(struct bulkdata *bulkdata, int p_idx)
{
	struct uci_section *s = NULL;
	int idx = 0, param_count = 0;

	buci_foreach_section("bulkdata", "profile_parameter", s) {
		char *profile_name = buci_get_value_bysection(s, "dm_parent");
		if (strcmp(bulkdata->profile[p_idx].profile_name, profile_name) != 0)
			continue;
		param_count++;
	}

	if (param_count == 0)
		return;

	bulkdata->profile[p_idx].profile_parameter = calloc(param_count, sizeof(struct profile_parameter));
	if (bulkdata->profile[p_idx].profile_parameter == NULL) {
		ERR("Failed to allocate memory for profile_parameter");
		return;
	}

	s = NULL;
	buci_foreach_section("bulkdata", "profile_parameter", s) {
		char *profile_name = buci_get_value_bysection(s, "dm_parent");
		if (strcmp(bulkdata->profile[p_idx].profile_name, profile_name) != 0)
			continue;

		load_profile_parameter_config(bulkdata, s, p_idx, idx);
		idx++;
	}

	bulkdata->profile[p_idx].profile_parameter_number = param_count;
}

static void load_profile_http_request_uri_parameter_config(struct bulkdata *bulkdata, struct uci_section *s, int p_idx, int idx)
{
	char *data = NULL;

	bulkdata->profile[p_idx].profile_http_uri_parameter[idx].value = NULL;
	bulkdata->profile[p_idx].profile_http_uri_parameter[idx].req = NULL;

	data = buci_get_value_bysection(s, "name");
	bulkdata->profile[p_idx].profile_http_uri_parameter[idx].name = strdup((STRLEN(data)) ? data : "");
	DEBUG("The HTTP resuest URI parameter name %d of profile name '%s' is : %s", idx+1, bulkdata->profile[p_idx].profile_name, bulkdata->profile[p_idx].profile_http_uri_parameter[idx].name);

	data = buci_get_value_bysection(s, "reference");
	bulkdata->profile[p_idx].profile_http_uri_parameter[idx].reference = strdup((STRLEN(data)) ? data : "");
	DEBUG("The HTTP resuest URI parameter reference %d of profile name '%s' is : %s", idx+1, bulkdata->profile[p_idx].profile_name, bulkdata->profile[p_idx].profile_http_uri_parameter[idx].reference);
}

static void get_profile_http_request_uri_parameter(struct bulkdata *bulkdata, int p_idx)
{
	struct uci_section *s = NULL;
	int idx = 0, uri_param_count = 0;

	buci_foreach_section("bulkdata", "profile_http_request_uri_parameter", s) {
		char *profile_name = buci_get_value_bysection(s, "dm_parent");
		if (strcmp(bulkdata->profile[p_idx].profile_name, profile_name) != 0)
			continue;

		uri_param_count++;
	}

	if (uri_param_count == 0)
		return;

	bulkdata->profile[p_idx].profile_http_uri_parameter = calloc(uri_param_count, sizeof(struct profile_http_request_uri_parameter));
	if (bulkdata->profile[p_idx].profile_http_uri_parameter == NULL) {
		ERR("Failed to allocate memory for http_uri_parameter");
		return;
	}

	s = NULL;
	buci_foreach_section("bulkdata", "profile_http_request_uri_parameter", s) {
		char *profile_name = buci_get_value_bysection(s, "dm_parent");
		if (strcmp(bulkdata->profile[p_idx].profile_name, profile_name) != 0)
			continue;

		load_profile_http_request_uri_parameter_config(bulkdata, s, p_idx, idx);
		idx++;
	}

	bulkdata->profile[p_idx].profile_http_request_uri_parameter_number = uri_param_count;
}

static void get_profiles_enable(struct bulkdata *bulkdata, struct device_id *device_id)
{
	struct uci_section *s = NULL;
	int profile_number = 0;

	bulkdata->profile_number = 0;

	buci_init();

	buci_foreach_section("bulkdata", "profile", s) {
		char *enable = buci_get_value_bysection(s, "enable");
		if (strcmp(enable, "1") == 0 || strcmp(enable, "true") == 0)
			profile_number++;
	}

	if (profile_number == 0) {
		buci_fini();
		return;
	}

	bulkdata->profile = calloc(profile_number, sizeof(struct profile));
	if (bulkdata->profile == NULL) {
		ERR("Failed to allocate memory for profile");
		buci_fini();
		return;
	}

	int idx = 0;
	buci_foreach_section("bulkdata", "profile", s) {
		char *enable = buci_get_value_bysection(s, "enable");
		if (strcmp(enable, "1") == 0 || strcmp(enable, "true") == 0) {
			load_profile_config(bulkdata, s, idx, device_id);
			get_profiles_parameters(bulkdata, idx);
			get_profile_http_request_uri_parameter(bulkdata, idx);
			bulkdata->profile[idx].failed_reports = calloc(1, sizeof(struct list_head));
			if (bulkdata->profile[idx].failed_reports != NULL) {
				INIT_LIST_HEAD(bulkdata->profile[idx].failed_reports);
			} else {
				ERR("Failed to allocate memory for profile failed reports");
			}
			idx++;
		}
	}

	bulkdata->profile_number = profile_number;

	buci_fini();
}

static void free_profiles_enable(struct bulkdata *bulkdata)
{
	for (int i = 0; i < bulkdata->profile_number; i++) {
		FREE(bulkdata->profile[i].profile_name);
		FREE(bulkdata->profile[i].protocol);
		FREE(bulkdata->profile[i].encoding_type);
		FREE(bulkdata->profile[i].csv_encoding_field_separator);
		FREE(bulkdata->profile[i].csv_encoding_row_separator);
		FREE(bulkdata->profile[i].csv_encoding_escape_character);
		FREE(bulkdata->profile[i].csv_encoding_report_format);
		FREE(bulkdata->profile[i].csv_encoding_row_time_stamp);
		FREE(bulkdata->profile[i].json_encoding_report_format);
		FREE(bulkdata->profile[i].json_encoding_report_time_stamp);
		FREE(bulkdata->profile[i].http_url);
		FREE(bulkdata->profile[i].http_username);
		FREE(bulkdata->profile[i].http_password);
		FREE(bulkdata->profile[i].http_compression);
		FREE(bulkdata->profile[i].http_method);
		FREE(bulkdata->profile[i].new_report);
		empty_failed_reports_list(&bulkdata->profile[i]);
		FREE(bulkdata->profile[i].failed_reports);
	}
	FREE(bulkdata->profile);
}

static void free_profiles_parameters(struct bulkdata *bulkdata)
{
	for (int i = 0; i < bulkdata->profile_number; i++) {
		for (int j = 0; j < bulkdata->profile[i].profile_parameter_number; j++) {
			FREE(bulkdata->profile[i].profile_parameter[j].name);
			FREE(bulkdata->profile[i].profile_parameter[j].reference);
			bulkdata_free_data_from_list(bulkdata->profile[i].profile_parameter[j].results);
			FREE(bulkdata->profile[i].profile_parameter[j].results);
		}
		FREE(bulkdata->profile[i].profile_parameter);
	}
}

static void free_profile_http_request_uri_parameter(struct bulkdata *bulkdata)
{
	for (int i = 0; i < bulkdata->profile_number; i++) {
		for (int j = 0; j < bulkdata->profile[i].profile_http_request_uri_parameter_number; j++) {
			FREE(bulkdata->profile[i].profile_http_uri_parameter[j].name);
			FREE(bulkdata->profile[i].profile_http_uri_parameter[j].reference);
			FREE(bulkdata->profile[i].profile_http_uri_parameter[j].value);
		}
		FREE(bulkdata->profile[i].profile_http_uri_parameter);
	}
}

void bulkdata_config_init(struct bulkdata *bulkdata, struct device_id *device_id)
{
	get_bulkdata_enable(bulkdata);
	get_profiles_enable(bulkdata, device_id);
}

void bulkdata_config_fini(struct bulkdata *bulkdata)
{
	free_profiles_parameters(bulkdata);
	free_profile_http_request_uri_parameter(bulkdata);
	free_profiles_enable(bulkdata);
	memset(bulkdata, 0, sizeof(struct bulkdata));
}
