/*
 * bulkdata.c - Bulkdata core
 *
 * Copyright (C) 2022-2024, IOPSYS Software Solutions AB.
 *
 * Author: Amin Ben Romdhane <amin.benromdhane@iopsys.eu>
 * Author: Omar Kallel <omar.kallel@pivasoftware.com>
 *
 * See LICENSE file for license related information.
 *
 */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <libubox/uloop.h>
#include <libubus.h>
#include <libbbfdm-ubus/bbfdm-ubus.h>
#include <libbbfdm-api/dmcommon.h>

#include "config.h"
#include "utils.h"

extern DM_MAP_OBJ tDynamicObj[];

struct bbfdm_context bbfdm_ctx = {0};
static struct bulkdata bulkdata_main = {0};
static struct ubus_event_handler ev = {0};
static struct device_id device_id = {0};
static void get_device_info(void);
static void bulkdata_profile_cb(struct uloop_timeout *timeout);

static void bulkdata_run_profiles(struct bulkdata *bulkdata)
{
	if (bulkdata->enable == false) {
		return;
	}

	for (int i = 0; i < bulkdata->profile_number; i++) {
		bulkdata->profile[i].utimer.cb = bulkdata_profile_cb;
		
		unsigned int next_period = get_next_period(bulkdata->profile[i].time_reference, bulkdata->profile[i].reporting_interval);
		INFO("The session of profile name '%s' will be start in %d sec", bulkdata->profile[i].profile_name, next_period);
		uloop_timeout_set(&bulkdata->profile[i].utimer, next_period * 1000);
	}
}

static void bulkdata_stop_profiles(struct bulkdata *bulkdata)
{
	for (int i = 0; i < bulkdata->profile_number; i++) {
		uloop_timeout_cancel(&bulkdata->profile[i].utimer);
		uloop_timeout_cancel(&bulkdata->profile[i].ctimer);

		// if any async req pending cancel the request
		cancel_pending_async_requests(&bulkdata->profile[i], false);
	}
}

static void bulkdata_generate_report_cb(struct uloop_timeout *timeout)
{
	struct profile *profile = NULL;

	profile = container_of(timeout, struct profile, ctimer);
	if (profile == NULL)
		return;

	// If any async call is still pending delete the requests
	cancel_pending_async_requests(profile, true);

	bulkdata_generate_report(profile);
}

static void bulkdata_profile_cb(struct uloop_timeout *timeout)
{
	struct profile *profile = NULL;
	int i;

	profile = container_of(timeout, struct profile, utimer);
	if (profile == NULL)
		return;

	INFO("New session of profile name '%s' started", profile->profile_name);
	profile->collection_time = time(NULL);

	for (i = 0; i < profile->profile_parameter_number; i++) {
		bulkdata_free_data_from_list(profile->profile_parameter[i].results);
		get_value_group(bbfdm_ctx.ubus_ctx, profile, i);
	}

	for (i = 0; i < profile->profile_http_request_uri_parameter_number; i++) {
		FREE(profile->profile_http_uri_parameter[i].value);

		if (profile->profile_http_uri_parameter[i].reference == NULL ||
		    strlen(profile->profile_http_uri_parameter[i].reference) == 0 ||
		    profile->profile_http_uri_parameter[i].name == NULL ||
		    strlen(profile->profile_http_uri_parameter[i].name) == 0)
			continue;

		get_value_single(bbfdm_ctx.ubus_ctx, profile, i);
	}

	profile->ctimer.cb = bulkdata_generate_report_cb;
	uloop_timeout_set(&profile->ctimer, 60000); // 60 seconds

	return;
}

static void bulkdata_get_deviceid_cb(struct uloop_timeout *timeout)
{
	get_device_info();
}

static void get_device_info(void)
{
	free_device_id_config(&device_id);
	get_device_id_config(bbfdm_ctx.ubus_ctx, &device_id);

	if (strlen(device_id.manufacturer_oui) == 0 && strlen(device_id.product_class) == 0 &&
	    strlen(device_id.serial_number) == 0) {
		device_id.retries++;

		if (device_id.retries < 5) {
			device_id.dtimer.cb = bulkdata_get_deviceid_cb;
			uloop_timeout_set(&device_id.dtimer, 5000);
		}
	} else {
		replace_spaces_with_urichar(&device_id);
	}
}

static void config_reload_cb(struct ubus_context *ctx __attribute__((unused)),
			struct ubus_event_handler *ev __attribute__((unused)),
			const char *type __attribute__((unused)),
			struct blob_attr *msg __attribute__((unused)))
{
	INFO("Reloading bulkdata upon bulkdata.reload event");
	bulkdata_stop_profiles(&bulkdata_main);
	bulkdata_config_fini(&bulkdata_main);
	bulkdata_config_init(&bulkdata_main, &device_id);
	bulkdata_run_profiles(&bulkdata_main);
}

static int register_config_change(struct ubus_context *uctx)
{
	int ret;

	memset(&ev, 0, sizeof(struct ubus_event_handler));
	ev.cb = config_reload_cb;

	ret = ubus_register_event_handler(uctx, &ev, "bulkdata.reload");
	if (ret)
		return -1;

	return 0;
}

static void usage(char *prog)
{
	fprintf(stderr, "Usage: %s [options]\n", prog);
	fprintf(stderr, "\n");
	fprintf(stderr, "options:\n");
	fprintf(stderr, "    -l <log level>  As per syslog, 0 to 7\n");
	fprintf(stderr, "    -d <schema dm>  Display the schema data model supported by micro-service\n");
	fprintf(stderr, "    -h <help>       To print this help menu\n");
	fprintf(stderr, "\n");
}

int main(int argc, char *argv[])
{
	int ch, dm_type = 0;
	int log_level = 3;

	while ((ch = getopt(argc, argv, "hdl:")) != -1) {
		switch (ch) {
		case 'l':
			log_level = (int)strtoul(optarg, NULL, 10);
			if (log_level < 0 || log_level > 7)
				log_level = 7;
			break;
		case 'd':
			dm_type++;
			break;
		case 'h':
			usage(argv[0]);
			exit(0);
		default:
			break;
		}
	}
	/* Logging to syslog */
	openlog("bulkdata", LOG_CONS | LOG_PID | LOG_NDELAY, LOG_LOCAL1);

	memset(&bbfdm_ctx, 0, sizeof(struct bbfdm_context));
	bbfdm_ubus_set_service_name(&bbfdm_ctx, "bulkdata");
	bbfdm_ubus_set_log_level(log_level);
	bbfdm_ubus_load_data_model(tDynamicObj);

	if (dm_type > 0) {
		int res = bbfdm_print_data_model_schema(&bbfdm_ctx, dm_type);
		exit(res);
	}

	if (bbfdm_ubus_register_init(&bbfdm_ctx))
		goto out;

	if (register_config_change(bbfdm_ctx.ubus_ctx) != 0)
		goto out;

	get_device_info();

	bulkdata_config_init(&bulkdata_main, &device_id);

	bulkdata_run_profiles(&bulkdata_main);
	uloop_run();

out:
	bbfdm_ubus_register_free(&bbfdm_ctx);
	free_device_id_config(&device_id);
	bulkdata_config_fini(&bulkdata_main);
	closelog();
	return 0;
}
