/*
 * utils.c - common utility functions
 *
 * Copyright (C) 2022-2024, IOPSYS Software Solutions AB.
 *
 * Author: Amin Ben Romdhane <amin.benromdhane@iopsys.eu>
 *
 * See LICENSE file for license related information.
 *
 */

#ifndef __UTILS_H
#define __UTILS_H

#include <syslog.h>
#include <libubus.h>
#include <libubox/uloop.h>
#include "config.h"

#include <libbbfdm-api/dmcommon.h>

#define DEFAULT_LOG_LEVEL 3

enum {
	HTTP_URI_PARAM_REF,
	PROFILE_PARAM_REF,
	__REF_MAX
};

typedef struct {
	struct list_head list;
	char *name;
	char *data;
	char *type;
} param_data_t;

typedef struct {
	struct profile *profile;
	int param_index;
} async_req_t;

struct failed_reports {
	struct list_head list;
	char *freport;
};

void create_encoding_bulkdata_report(struct profile *profile, char **report);
void create_failed_report(struct profile *profile, char **report);
void bulkdata_generate_report(struct profile *profile);

void get_time_stamp(const char *format, time_t time, char **timestamp);
unsigned int get_next_period(time_t time_reference, int reporting_interval);

void bulkdata_free_data_from_list(struct list_head *dup_list);

int get_value_single(struct ubus_context *ubus_ctx, struct profile *profile, int param_index);
int get_value_group(struct ubus_context *ubus_ctx, struct profile *profile, int param_index);
void get_device_id_config(struct ubus_context *ctx, struct device_id *device_id);
void free_device_id_config(struct device_id *device_id);

int http_send_message(struct profile *profile, char *msg_out);

bool bulkdata_get_value(struct ubus_context *ctx, const char *parameter_name, void *value);
char *get_bulkdata_profile_parameter_name(const char *paramref, const char *paramname, const char *param);
void append_string_to_string(const char *strappend, char **target);
void bulkdata_add_failed_report(struct profile *profile, const char *freport);

struct failed_reports *empty_failed_reports_list(struct profile *profile);
void add_failed_reports_to_report_csv(struct profile *profile, char **report, int isnext);

int wait_for_bbfdm_object(void);
void cancel_pending_async_requests(struct profile *profile, bool log_failed);
void replace_spaces_with_urichar(struct device_id *device_id);

void print_info(const char *format, ...);
void print_debug(const char *format, ...);
void print_error(const char *format, ...);
void print_warning(const char *format, ...);
int get_log_level_config(void);

void safe_asprintf(char **strp, const char *fmt, ...);

#define ERR(fmt, args...) \
	BBF_ERR(fmt, #args)

#define WARNING(fmt, args...) \
	BBF_WARNING(fmt, #args)

#define INFO(fmt, args...) \
	BBF_INFO(fmt, #args)

#define DEBUG(fmt, args...) \
	BBF_DEBUG(fmt, #args)

#ifndef STRLEN
#define STRLEN(SRC) ((SRC != NULL) ? strlen(SRC) : 0)
#endif

#ifndef STRNCPY
#define STRNCPY(DST, SRC, SIZE) \
do { \
	if ((DST) != NULL && (SRC) != NULL) { \
		strncpy(DST, SRC, SIZE - 1); \
		DST[SIZE-1] = '\0'; \
	} \
} while(0)
#endif

#ifndef FREE
#define FREE(x) do { if (x) {free(x); x = NULL;} } while (0)
#endif

#endif /* __UTILS_H */
