/*
 * report.c - bulkdata report generation
 *
 * Copyright (C) 2022-2024, IOPSYS Software Solutions AB.
 *
 * Author: Amin Ben Romdhane <amin.benromdhane@iopsys.eu>
 * Author: Omar Kallel <omar.kallel@pivasoftware.com>
 *
 * See LICENSE file for license related information.
 *
 */

#include <string.h>
#include <fcntl.h>
#include <json-c/json.h>

#include "config.h"
#include "utils.h"

static int get_retry_period(int min)
{
	unsigned int random_value = 0;

	int fd = open("/dev/urandom", O_RDONLY);
	if (fd < 0) {
		return min;
	}

	if (read(fd, &random_value, sizeof(random_value)) != sizeof(random_value)) {
		close(fd);
		return min;
	}

	close(fd);

	// Generate a random number in the range [min, 2*min)
	return (random_value % min) + min;
}

static void add_new_json_obj(json_object *json_obj, const char *name, const char *data, const char *type)
{
	json_object *jobj;
	if (strstr(type, "unsignedInt") || strstr(type, "int") || strstr(type, "long"))
		jobj = json_object_new_int64((int64_t)strtol(data, NULL, 10));
	else if (strstr(type, "bool"))
		jobj = json_object_new_boolean((int64_t)strtol(data, NULL, 10));
	else
		jobj = json_object_new_string(data);

	json_object_object_add(json_obj, name, jobj);
}

static json_object *json_object_select_obj(json_object *jobj, char *argv[])
{
	int i;
	for (i = 0; argv[i]; i++) {
		if (jobj == NULL)
			return NULL;
		json_object_object_get_ex(jobj, argv[i], &jobj);
	}
	return jobj;
}

static void create_json_bulkdata_report_object_hierarchy(struct profile *profile, char **report)
{
	int i, j, profile_param_number = profile->profile_parameter_number;
	char *param_name, *result, *pch, *pchr, *collection_time = NULL;
	char buf[256] = {0};

	struct json_object *json_obj = json_object_new_object();

	get_time_stamp(profile->json_encoding_report_time_stamp, profile->collection_time, &collection_time);
	if (collection_time) {
		if (strcasecmp(profile->json_encoding_report_time_stamp, "ISO-8601") == 0)
			json_object_object_add(json_obj, "CollectionTime", json_object_new_string(collection_time));
		else
			json_object_object_add(json_obj, "CollectionTime", json_object_new_int64((int64_t)strtol(collection_time, NULL, 10)));
		free(collection_time);
	}

	struct json_object *json_obj2 = json_obj;
	for (i = 0; i < profile_param_number; i++) {
		param_data_t *p = NULL, *tmp = NULL;
		struct list_head *result_list = profile->profile_parameter[i].results;
		if (result_list == NULL || list_empty(result_list))
			continue;

		list_for_each_entry_safe(p, tmp, result_list, list) {
			char *argv[128] = {0};
			j = 0;

			param_name = get_bulkdata_profile_parameter_name(profile->profile_parameter[i].reference, profile->profile_parameter[i].name, p->name);

			STRNCPY(buf, param_name, sizeof(buf));
			for (pch = strtok_r(buf, ".", &pchr); pch != NULL; pch = strtok_r(NULL, ".", &pchr)) {
				argv[j] = pch;
				struct json_object *json_obj1 = json_object_select_obj(json_obj, argv);
				if (json_obj1)
					json_obj2 = json_obj1;
				else {
					if (pchr != NULL && *pchr != '\0') {
						json_object *new_obj = json_object_new_object();
						json_object_object_add(json_obj2, pch, new_obj);
						json_obj2 = new_obj;
					}
					else
						add_new_json_obj(json_obj2, pch, p->data, p->type);
				}
				j++;
			}

			FREE(param_name);
		}

		bulkdata_free_data_from_list(result_list);
	}

	result = (char *)json_object_to_json_string_ext(json_obj, JSON_C_TO_STRING_PRETTY);
	*report = strdup(result);
	json_object_put(json_obj);
}

static void create_json_bulkdata_report_name_value_pair(struct profile *profile, char **report)
{
	struct json_object *json_obj = NULL;
	char *param_name, *result, *collection_time = NULL;
	int i = 0, profile_param_number = profile->profile_parameter_number;

	json_obj = json_object_new_object();
	get_time_stamp(profile->json_encoding_report_time_stamp, profile->collection_time, &collection_time);
	if (collection_time) {
		if (strcasecmp(profile->json_encoding_report_time_stamp, "ISO-8601") == 0)
			json_object_object_add(json_obj, "CollectionTime", json_object_new_string(collection_time));
		else
			json_object_object_add(json_obj, "CollectionTime", json_object_new_int64((int64_t)strtol(collection_time, NULL, 10)));
		free(collection_time);
	}

	for (i = 0; i < profile_param_number; i++) {
		param_data_t *p = NULL, *tmp = NULL;
		struct list_head *result_list = profile->profile_parameter[i].results;
		if (result_list == NULL || list_empty(result_list))
			continue;

		list_for_each_entry_safe(p, tmp, result_list, list) {
			param_name = get_bulkdata_profile_parameter_name(profile->profile_parameter[i].reference, profile->profile_parameter[i].name, p->name);
			add_new_json_obj(json_obj, param_name, p->data, p->type);
			FREE(param_name);
		}

		bulkdata_free_data_from_list(result_list);
	}

	result = (char *)json_object_to_json_string_ext(json_obj, JSON_C_TO_STRING_PRETTY);
	*report = strdup(result);
	json_object_put(json_obj);
}

static void add_failed_reports_to_report_json(struct profile *profile, char *new_report, char **report, int isnext)
{
	json_object *json_obj, *json_array, *json_string;
	struct failed_reports *retreport = NULL;
	char *msgout = NULL;
	int j = 0;

	if (profile->failed_reports == NULL)
		return;

	json_obj = json_object_new_object();
	json_array = json_object_new_array();
	json_object_object_add(json_obj,"Report", json_array);

	if (list_empty(profile->failed_reports))
		goto new_report;

	list_for_each_entry(retreport, profile->failed_reports, list) {
		if (!j && isnext) {
			j = 1;
			continue;
		}
		json_string = json_tokener_parse(retreport->freport);
		json_object_array_add(json_array, json_string);
	}

new_report :
	if (new_report) {
		json_string = json_tokener_parse(new_report);
		json_object_array_add(json_array, json_string);
	}

	msgout = (char *)json_object_to_json_string_ext(json_obj, JSON_C_TO_STRING_PRETTY);
	*report = strdup(msgout);
	json_object_put(json_obj);
}

static void create_report_json(char *new_report, char **report)
{
	json_object *json_array = NULL;
	json_object *json_obj = NULL;
	char *msgout = NULL;

	json_obj = json_object_new_object();
	json_array = json_object_new_array();
	json_object_object_add(json_obj,"Report", json_array);

	if (new_report) {
		json_object *json_string = NULL;

		json_string = json_tokener_parse(new_report);
		json_object_array_add(json_array, json_string);
	}

	msgout = (char *)json_object_to_json_string_ext(json_obj, JSON_C_TO_STRING_PRETTY);
	*report = strdup(msgout);
	json_object_put(json_obj);
}

static char *csv_encode(const char *str)
{
	if (str == NULL)
		return NULL;

	int len = strlen(str);
	if (len == 0)
		return strdup(str);

	char *temp = NULL;
	// Get the number of '\"' present in the string
	int quote_count = 0;

	temp = strchr(str, '\"');
	while (temp) {
		quote_count++;
		temp = strchr(temp+1, '\"');
	}

	int encode_size = len + quote_count + 3; // added 3 for initial quote, end quote & null at end
	temp = (char *)malloc(sizeof(char) * encode_size);

	if (!temp)
		return NULL;

	memset(temp, 0, sizeof(char) * encode_size);

	int i = 0, j = 0;

	temp[j++] = '\"';
	for (i = 0; i < len; i++) {
		if (str[i] == '\"') {
			if (j > (encode_size - 3))
				break;

			temp[j++] = '\"';
		}

		if (j > (encode_size - 3))
			break;

		temp[j++] = str[i];
	}

	temp[j] = '\"';

	return temp;
}

int create_json_bulkdata_report(struct profile *profile, char **report)
{
	/*
	 * create json msg of current report
	 * parse failed reports list and add it to the report
	 * then add new report to the report
	 */
	char *msgout = NULL;

	profile->new_report = NULL;
	if (strcasecmp(profile->json_encoding_report_format, "ObjectHierarchy") == 0) {
		create_json_bulkdata_report_object_hierarchy(profile, &msgout);
	} else if (strcasecmp(profile->json_encoding_report_format, "NameValuePair") == 0) {
		create_json_bulkdata_report_name_value_pair(profile, &msgout);
	}

	if (profile->nbre_of_retained_failed_reports != 0) {
		if (profile->nbre_failed_reports >= profile->nbre_of_retained_failed_reports && profile->nbre_of_retained_failed_reports > 0)
			add_failed_reports_to_report_json(profile, msgout, report, 1);
		else
			add_failed_reports_to_report_json(profile, msgout, report, 0);
	} else {
		create_report_json(msgout, report);
	}

	append_string_to_string(msgout, &profile->new_report);
	FREE(msgout);
	return 0;
}

int create_csv_bulkdata_report(struct profile *profile, char **report)
{
	/*
	 * create csv msg of current report
	 * parse failed reports list and add it to the report
	 */
	int i;
	char *str1 = NULL, *str2 = NULL, *str = NULL, *paramprofilename, *timestamp = NULL, *type = NULL, *val = NULL, rowseparator = '\0', separator = '\0';

	if (strcmp(profile->csv_encoding_row_separator, "&#10;") == 0)
		rowseparator = '\n';
	else if (strcmp(profile->csv_encoding_row_separator, "&#13;") == 0)
		rowseparator = '\r';

	if (profile->csv_encoding_field_separator)
		separator = profile->csv_encoding_field_separator[0];

	get_time_stamp(profile->csv_encoding_row_time_stamp, profile->collection_time, &val);
	timestamp = csv_encode(val);
	FREE(val);

	/*
	 * Create header ReportTimestamp,ParameterName,ParameterValue,ParameterType in case of ParameterPerRow
	 */
	if (strcasecmp(profile->csv_encoding_report_format, "ParameterPerRow") == 0) {
		if (timestamp == NULL)
			safe_asprintf(&str, "ParameterName%cParameterValue%cParameterType%c", separator, separator, rowseparator);
		else
			safe_asprintf(&str, "ReportTimestamp%cParameterName%cParameterValue%cParameterType%c", separator, separator, separator, rowseparator);
		append_string_to_string(str, report);
		FREE(str);
		if (profile->nbre_of_retained_failed_reports != 0) {
			if (profile->nbre_failed_reports >= profile->nbre_of_retained_failed_reports && profile->nbre_of_retained_failed_reports > 0)
				add_failed_reports_to_report_csv(profile, report, 1);
			else
				add_failed_reports_to_report_csv(profile, report, 0);
		}
	}

	if (strcasecmp(profile->csv_encoding_report_format, "ParameterPerColumn") == 0 && timestamp != NULL) {
		if (profile->nbre_of_retained_failed_reports != 0) {
			if (profile->nbre_failed_reports >= profile->nbre_of_retained_failed_reports && profile->nbre_of_retained_failed_reports > 0)
				add_failed_reports_to_report_csv(profile, report, 1);
			else
				add_failed_reports_to_report_csv(profile, report, 0);
		}
		append_string_to_string("ReportTimestamp", &str1);
		append_string_to_string(timestamp, &str2);
	}

	/*
	 * Add New reports
	 */
	profile->new_report = NULL;
	for(i = 0; i < profile->profile_parameter_number; i++) {
		struct list_head *result_list = profile->profile_parameter[i].results;
		if (result_list == NULL || list_empty(result_list))
			continue;

		param_data_t *p = NULL, *tmp = NULL;

		list_for_each_entry_safe(p, tmp, result_list, list) {
			paramprofilename = get_bulkdata_profile_parameter_name(profile->profile_parameter[i].reference, profile->profile_parameter[i].name, p->name);
			char *p_path = csv_encode(paramprofilename);
			char *p_value = csv_encode(p->data);
			if (strcasecmp(profile->csv_encoding_report_format, "ParameterPerRow") == 0) {
				type = strstr(p->type, ":");
				if (timestamp == NULL)
					safe_asprintf(&str, "%s%c%s%c%s%c", p_path, separator, p_value, separator, type+1, rowseparator);
				else
					safe_asprintf(&str, "%s%c%s%c%s%c%s%c", timestamp, separator, p_path, separator, p_value, separator, type+1, rowseparator);
				append_string_to_string(str, report);
				append_string_to_string(str, &profile->new_report);
				FREE(str);
			} else if (strcasecmp(profile->csv_encoding_report_format, "ParameterPerColumn") == 0) {
				if (str1 == NULL || strlen(str1) == 0)
					safe_asprintf(&str, "%s", p_path);
				else
					safe_asprintf(&str, "%c%s", separator, p_path);
				append_string_to_string(str, &str1);
				FREE(str);
				if (str2 == NULL || strlen(str2) == 0)
					safe_asprintf(&str, "%s", p_value);
				else
					safe_asprintf(&str, "%c%s", separator, p_value);
				append_string_to_string(str, &str2);
				FREE(str);
			}
			FREE(paramprofilename);
			FREE(p_path);
			FREE(p_value);
		}

		bulkdata_free_data_from_list(result_list);
	}
	if (strcasecmp(profile->csv_encoding_report_format, "ParameterPerColumn") == 0) {
		safe_asprintf(&str, "%c", rowseparator);
		append_string_to_string(str, &str1);
		append_string_to_string(str, &str2);
		append_string_to_string(str1, report);
		append_string_to_string(str2, report);
		append_string_to_string(str1, &profile->new_report);
		append_string_to_string(str2, &profile->new_report);
	}
	FREE(str);
	FREE(str1);
	FREE(str2);
	FREE(timestamp);
	return 0;
}

static void create_json_failed_report(struct profile *profile, char **report)
{
	add_failed_reports_to_report_json(profile, NULL, report, 0);
}

static void create_csv_failed_report(struct profile *profile, char **report)
{
	char rowseparator = '\0', separator = '\0';

	if (strcmp(profile->csv_encoding_row_separator, "&#10;") == 0) {
		rowseparator = '\n';
	} else if (strcmp(profile->csv_encoding_row_separator, "&#13;") == 0) {
		rowseparator = '\r';
	}

	if (profile->csv_encoding_field_separator)
		separator = profile->csv_encoding_field_separator[0];

	if (strcasecmp(profile->csv_encoding_report_format, "ParameterPerRow") == 0) {
		char *timestamp = NULL;

		get_time_stamp(profile->csv_encoding_row_time_stamp, profile->collection_time, &timestamp);

		if (timestamp == NULL) {
			safe_asprintf(report, "ParameterName%cParameterValue%cParameterType%c", separator, separator, rowseparator);
		} else {
			safe_asprintf(report, "ReportTimestamp%cParameterName%cParameterValue%cParameterType%c", separator, separator, separator, rowseparator);
			FREE(timestamp);
		}
	}
	add_failed_reports_to_report_csv(profile, report, 0);
}

void create_encoding_bulkdata_report(struct profile *profile, char **report)
{
	if (strcasecmp(profile->encoding_type, "JSON") == 0) {
		create_json_bulkdata_report(profile, report);
	} else if (strcasecmp(profile->encoding_type, "CSV") == 0) {
		create_csv_bulkdata_report(profile, report);
	}
}

void create_failed_report(struct profile *profile, char **report)
{
	if (strcasecmp(profile->encoding_type, "JSON") == 0) {
		create_json_failed_report(profile, report);
	} else if (strcasecmp(profile->encoding_type, "CSV") == 0) {
		create_csv_failed_report(profile, report);
	}
}

void bulkdata_generate_report(struct profile *profile)
{
	char *report = NULL;
	unsigned int http_code = 0;

	if (profile == NULL)
		return;

	// Start generating reports
	time_t now = profile->collection_time;
	if (profile->retry_count == 0 || profile->next_retry > now || !profile->http_retry_enable) //Perdiodic execution
		create_encoding_bulkdata_report(profile, &report);
	else
		create_failed_report(profile, &report);

	DEBUG("The content of the profile name report '%s' is :\n==========\n%s\n==========\n", profile->profile_name, report);
	http_code = http_send_message(profile, report);
	if (http_code != 200) {
		unsigned int retry_period = 0;

		if (profile->retry_count == 0 || profile->next_retry > now || !profile->http_retry_enable) {  //Perdiodic execution
			retry_period = get_retry_period(profile->http_retry_minimum_wait_interval);
			profile->next_period = now + profile->reporting_interval;
			profile->next_retry = now + retry_period;
			profile->retry_count = 1;
			profile->min_retry = profile->http_retry_minimum_wait_interval * 2;
			if ((profile->next_retry < profile->next_period) && profile->http_retry_enable) {
				INFO("Retry session of profile name '%s' in %d sec", profile->profile_name, retry_period);
				uloop_timeout_set(&profile->utimer, 1000 * retry_period);
			} else {
				INFO("Start New session of profile name '%s' in %d sec", profile->profile_name, profile->reporting_interval);
				uloop_timeout_set(&profile->utimer, 1000 * profile->reporting_interval);
			}
		} else { //Retry execution
			retry_period = get_retry_period(profile->min_retry);
			profile->min_retry*=2;
			profile->next_retry+=retry_period;
			profile->retry_count++;
			if (profile->next_retry < profile->next_period) {
				INFO("Retry session of profile name '%s' in %d sec", profile->profile_name, retry_period);
				uloop_timeout_set(&profile->utimer, 1000 * retry_period);
			} else {
				INFO("Retry session of profile name '%s' in %ld sec", profile->profile_name, (profile->next_period-profile->next_retry+retry_period));
				uloop_timeout_set(&profile->utimer, 1000 * (profile->next_period - profile->next_retry + retry_period));
			}
		}

		if (profile->new_report) {
			bulkdata_add_failed_report(profile, profile->new_report);
			FREE(profile->new_report);
		}
		FREE(report);
	} else {
		if (profile->retry_count == 0 || profile->next_retry > now || !profile->http_retry_enable) {
			INFO("Start New session of profile name '%s' in %d sec", profile->profile_name, profile->reporting_interval);
			uloop_timeout_set(&profile->utimer, 1000 * profile->reporting_interval);
		} else {
			INFO("Retry session of profile name '%s' in %ld sec", profile->profile_name, (profile->next_period-profile->next_retry));
			uloop_timeout_set(&profile->utimer, 1000 * (profile->next_period - profile->next_retry));
		}
		FREE(profile->new_report);
		FREE(report);
		empty_failed_reports_list(profile);
		profile->retry_count= 0;
	}
}
