/*
 * buci.c - uci utilities
 *
 * Copyright (C) 2021-2024, IOPSYS Software Solutions AB.
 *
 * Author: Amin Ben Romdhane <amin.benromdhane@iopsys.eu>
 *
 * See LICENSE file for license related information.
 *
 */

#include <strings.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <ctype.h>

#include "buci.h"

struct uci_context *uci_ctx = NULL;

int buci_init(void)
{
	uci_ctx = uci_alloc_context();
	if (!uci_ctx) {
		return -1;
	}
	return 0;
}

int buci_fini(void)
{
	if (uci_ctx) {
		uci_free_context(uci_ctx);
	}

	return 0;
}

static bool buci_validate_section(const char *str)
{
	if (!*str)
		return false;

	for (; *str; str++) {
		unsigned char c = *str;

		if (isalnum(c) || c == '_')
			continue;

		return false;
	}
	return true;
}

static int buci_init_ptr(struct uci_ptr *ptr, const char *package, const char *section, const char *option, const char *value)
{
	memset(ptr, 0, sizeof(struct uci_ptr));

	/* value */
	if (value) {
		ptr->value = value;
	}
	ptr->package = package;
	if (!ptr->package)
		goto error;

	ptr->section = section;
	if (!ptr->section) {
		ptr->target = UCI_TYPE_PACKAGE;
		goto lastval;
	}

	ptr->option = option;
	if (!ptr->option) {
		ptr->target = UCI_TYPE_SECTION;
		goto lastval;
	} else {
		ptr->target = UCI_TYPE_OPTION;
	}

lastval:
	if (ptr->section && !buci_validate_section(ptr->section))
		ptr->flags |= UCI_LOOKUP_EXTENDED;

	return 0;

error:
	return -1;
}

struct uci_section *buci_walk_section(const char *package, const char *section_type, struct uci_section *prev_section)
{
	struct uci_ptr ptr;
	struct uci_element *e;
	struct uci_section *next_section;

	if (section_type == NULL) {
		if (prev_section) {
			e = &prev_section->e;
			if (e->list.next == &prev_section->package->sections)
				return NULL;
			e = container_of(e->list.next, struct uci_element, list);
			next_section = uci_to_section(e);
			return next_section;
		}
		else {
			if (buci_init_ptr(&ptr, package, NULL, NULL, NULL))
				return NULL;

			if (uci_lookup_ptr(uci_ctx, &ptr, NULL, true) != UCI_OK)
				return NULL;

			if (ptr.p->sections.next == &ptr.p->sections)
				return NULL;

			e = container_of(ptr.p->sections.next, struct uci_element, list);
			next_section = uci_to_section(e);

			return next_section;
		}
	}
	else {
		struct uci_list *ul, *shead = NULL;

		if (prev_section) {
			ul = &prev_section->e.list;
			shead = &prev_section->package->sections;
		}
		else {
			if (buci_init_ptr(&ptr, package, NULL, NULL, NULL))
				return NULL;

			if (uci_lookup_ptr(uci_ctx, &ptr, NULL, true) != UCI_OK)
				return NULL;

			ul = &ptr.p->sections;
			shead = &ptr.p->sections;
		}
		while (ul->next != shead) {
			e = container_of(ul->next, struct uci_element, list);
			next_section = uci_to_section(e);
			if (strcmp(next_section->type, section_type) == 0)
				return next_section;
			ul = ul->next;
		}
		return NULL;
	}
	return NULL;
}

static struct uci_element *buci_lookup_list(struct uci_list *list, const char *name)
{
	struct uci_element *e;

	uci_foreach_element(list, e) {
		if (!strcmp(e->name, name))
			return e;
	}
	return NULL;
}

static int uci_lookup_ptr_bysection(struct uci_ptr *ptr, struct uci_section *section, const char *option, const char *value)
{
	struct uci_element *e;
	memset(ptr, 0, sizeof(struct uci_ptr));

	ptr->package = section->package->e.name;
	ptr->section = section->e.name;
	ptr->option = option;
	ptr->value = value;
	ptr->flags |= UCI_LOOKUP_DONE;

	ptr->p = section->package;
	ptr->s = section;

	if (ptr->option) {
		e = buci_lookup_list(&ptr->s->options, ptr->option);
		if (!e)
			return UCI_OK;
		ptr->o = uci_to_option(e);
		ptr->last = e;
		ptr->target = UCI_TYPE_OPTION;
	}
	else {
		ptr->last = &ptr->s->e;
		ptr->target = UCI_TYPE_SECTION;
	}

	ptr->flags |= UCI_LOOKUP_COMPLETE;

	return UCI_OK;
}

char *buci_get_value_bysection(struct uci_section *section, const char *option)
{
	struct uci_ptr ptr = {0};

	if (!section || !option)
		return "";

	if (uci_lookup_ptr_bysection(&ptr, section, option, NULL) != UCI_OK)
		return "";

	if (!ptr.o)
		return "";

	if (ptr.o->v.string)
		return ptr.o->v.string;
	else
		return "";
}

char *buci_get_value(const char *package, const char *section, const char *option)
{
	struct uci_ptr ptr = {0};

	if (!package || !section || !option)
		return "";

	if (buci_init_ptr(&ptr, package, section, option, NULL))
		return "";

	if (uci_lookup_ptr(uci_ctx, &ptr, NULL, true) != UCI_OK)
		return "";

	if (!ptr.o)
		return "";

	if (ptr.o->v.string)
		return ptr.o->v.string;
	else
		return "";
}

void buci_set_value_varstate(const char *uci_param, const char *value)
{
	struct uci_ptr ptr = {0};
	char uci_path[256] = {0};
	struct uci_context *uci_ctxt = NULL;

	if (!uci_param || !value)
		return;

	uci_ctxt = uci_alloc_context();
	if (!uci_ctxt) {
		return;
	}

	uci_set_confdir(uci_ctxt, "/var/state");

	snprintf(uci_path, sizeof(uci_path), "%s=%s", uci_param, value);
	if (uci_lookup_ptr(uci_ctxt, &ptr, uci_path, true) != UCI_OK) {
		uci_free_context(uci_ctxt);
		return;
	}

	if (uci_set(uci_ctxt, &ptr) != UCI_OK) {
		uci_free_context(uci_ctxt);
		return;
	}

	if (uci_save(uci_ctxt, ptr.p) != UCI_OK) {
		uci_free_context(uci_ctxt);
		return;
	}

	uci_commit(uci_ctxt, &ptr.p, false);
	uci_free_context(uci_ctxt);
	return;
}
