/*
 * Copyright (C) 2024 iopsys Software Solutions AB
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 2.1
 * as published by the Free Software Foundation
 *
 *		Author: Amin Ben Romdhane <amin.benromdhane@iopsys.eu>
 *		Author: Mohd Husaam Mehdi <husaam.mehdi@iopsys.eu>
 *
 */

#include "common.h"

/**************************************************************************
* COMMON FUNCTIONS
***************************************************************************/
static bool multi_ap_type_backhaul(struct uci_section *config_section)
{
	bool ret = false;
	char *multi_ap = NULL;

	 /* multi_ap              type
	  * 1                     Backhaul
	  * 2                     Fronthaul
	  * 3                     Combined
	  * Others                None
	  */
	dmuci_get_value_by_section_string(config_section, "multi_ap", &multi_ap);
	if (DM_STRCMP(multi_ap, "1") == 0)
		ret = true;

	return ret;
}

static struct uci_section *ethernet___get_ethernet_interface_section(const char *device_name)
{
	struct uci_section *s = NULL;

	uci_foreach_sections("network", "device", s) {
		char *name = NULL;

		if (!dmuci_is_option_value_empty(s, "type"))
			continue;

		dmuci_get_value_by_section_string(s, "name", &name);

		if (DM_STRCMP(name, device_name) == 0)
			return s;
	}

	return NULL;
}

static void remove_device_from_bridge_interface(struct uci_section *br_sec)
{
	struct uci_section *s = NULL, *stmp = NULL;
	char *device = NULL;

	if (!br_sec)
		return;

	dmuci_get_value_by_section_string(br_sec, "name", &device);
	if (DM_STRLEN(device) == 0 )
		return;

	uci_foreach_option_eq_safe("network", "interface", "device", device, stmp, s) {
		char *proto = NULL;

		dmuci_get_value_by_section_string(s, "proto", &proto);
		dmuci_delete_by_section(s, (proto && *proto == 0) ? NULL : "device", NULL);
		break;
	}
}

static void add_port_to_bridge_sections(struct uci_section *br_sec, struct uci_section *br_dmmap_sec, const char *device_port)
{
	struct uci_list *uci_list = NULL;

	if (DM_STRLEN(device_port) == 0)
		return;

	if (br_sec) {
		dmuci_get_value_by_section_list(br_sec, "ports", &uci_list);
		if (!value_exists_in_uci_list(uci_list, device_port))
			dmuci_add_list_value_by_section(br_sec, "ports", device_port);
	}

	if (br_dmmap_sec) {
		dmuci_get_value_by_section_list(br_dmmap_sec, "ports", &uci_list);
		if (!value_exists_in_uci_list(uci_list, device_port))
			dmuci_add_list_value_by_section(br_dmmap_sec, "ports", device_port);
	}
}

static void remove_port_from_bridge_sections(struct uci_section *br_sec, struct uci_section *br_dmmap_sec, const char *device_port)
{
	struct uci_list *uci_list = NULL;

	if (DM_STRLEN(device_port) == 0)
		return;

	if (br_sec) {
		dmuci_get_value_by_section_list(br_sec, "ports", &uci_list);
		if (value_exists_in_uci_list(uci_list, device_port))
			dmuci_del_list_value_by_section(br_sec, "ports", device_port);
	}

	if (br_dmmap_sec) {
		dmuci_get_value_by_section_list(br_dmmap_sec, "ports", &uci_list);
		if (value_exists_in_uci_list(uci_list, device_port))
			dmuci_del_list_value_by_section(br_dmmap_sec, "ports", device_port);
	}
}

static void get_8021q_port(char *dest_8021q_port, size_t port_str_size, char *port_path, const char *port_name)
{
	if (!DM_STRLEN(port_name)) {
		return;
	}

	if (!dest_8021q_port || !port_str_size) {
		return;
	}

	// either port_path or src_bridge_inst is needed to get src_bridge_inst
	if (!DM_STRLEN(port_path)) {
		return;
	}

	char bridge_inst[8] = {0};
	// get the bridge instance from port_path
	// Device.Bridging.Bridge.1.Port.2
	// the index of third dot is fixed at 23
	if (DM_STRLEN(port_path) > 22) {
		char *start = port_path + 22;
		char *end = NULL;

		end = DM_STRSTR(start + 1, ".");
		if (!end)
			return;

		// copy the bridge instance
		DM_STRNCPY(bridge_inst, start + 1, end - (start + 1));
	}

	if (!DM_STRLEN(bridge_inst)) {
		return;
	}

	struct uci_section *src_bridge_vlan_sec = NULL;
	src_bridge_vlan_sec = get_dup_section_in_dmmap_opt("dmmap_bridge_vlan", "bridge-vlan", "br_inst", bridge_inst);
	// get the vlan from here
	char *src_vlan = NULL;
	if (src_bridge_vlan_sec) {
		dmuci_get_value_by_section_string(src_bridge_vlan_sec, "vlan", &src_vlan);

		if (DM_STRLEN(src_vlan) == 0) {
			return;
		}
	} else {
		return;
	}

	// create 8021q section using this info
	struct uci_section *dest_8021q_sec = NULL;
	dmuci_add_section("network", "device", &dest_8021q_sec);
	dmuci_set_value_by_section(dest_8021q_sec, "type", "8021q");
	dmuci_set_value_by_section(dest_8021q_sec, "vid", src_vlan);
	dmuci_set_value_by_section(dest_8021q_sec, "ifname", port_name);

	snprintf(dest_8021q_port, port_str_size, "%s.%s", port_name, src_vlan);

	dmuci_set_value_by_section(dest_8021q_sec, "name", dest_8021q_port);

	// delete source bridge-vlan section
	if (src_bridge_vlan_sec) {
		char *src_bridge_vlan_sec_name = NULL;
		dmuci_get_value_by_section_string(src_bridge_vlan_sec, "section_name", &src_bridge_vlan_sec_name);

		if (DM_STRLEN(src_bridge_vlan_sec_name)) {
			struct uci_section *src_bridge_vlan_config_section = get_origin_section_from_config("network", "bridge-vlan", src_bridge_vlan_sec_name);
			if (src_bridge_vlan_config_section) {
				dmuci_delete_by_section(src_bridge_vlan_config_section, NULL, NULL);
			}
		}
	}
}

static void set_Provider_bridge_component(char *refparam, struct dmctx *ctx, void *data, char *instance, const char *linker, const char *component)
{
	/* *value=Device.Bridging.Bridge.{i}.
	 * In file dmmap_provider_bridge set "option svlan_br_inst {i}" or "list cvlan_br_inst {i}" in this(refered "provider_bridge" section)
	 */
	struct uci_section *s = NULL, *tmp_s = NULL, *dmmap_bridge_section = NULL;
	struct uci_section *network_bridge_sec_from = NULL, *network_bridge_sec_to = NULL;
	char pr_br_sec_name[64] = {0};
	char *br_sec_name = NULL;

	if (DM_STRLEN(linker) == 0) // Linker should be like "bridge-X"
		return;

	char *br_inst = DM_STRCHR(linker, '-'); // Get bridge instance 'X' which is linker from Name parameter 'bridge-X'
	if (!br_inst)
		return;

	// section name of bridge in network file
	snprintf(pr_br_sec_name, sizeof(pr_br_sec_name), "pr_br_%s", instance);

	/*
	 * check if provider bridge instance of this provider bridge is present in network uci file
	 * if present add candidate bridge to this provider bridge instance.
	 * if not present, create a provider bridge instance in network uci file,
	 * i.e. just update the candidate bridge section name to pr_br_{i} | {i} = instance of provider bridge
	 */
	uci_foreach_option_eq("network", "device", "type", "bridge", s) {
		if (strcmp(pr_br_sec_name, section_name(s)) == 0) {
			network_bridge_sec_to = s;
			break;
		}
	}

	if (DM_LSTRCMP(component, "CVLAN") == 0) {
		// Set svlan_br_inst in dmmap_provider_bridge->provider_bridge section

		dmuci_add_list_value_by_section(((struct dm_data *)data)->config_section, "cvlan_br_inst", br_inst + 1);
	} else if (DM_LSTRCMP(component, "SVLAN") == 0) {
		// Set svlan_br_inst in dmmap_provider_bridgei->provider_bridge section

		dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "svlan_br_inst", br_inst + 1);
	}

	/* Add candidate bridge to this provider bridge instance(network->device->pr_br_{i}) */
	// Get network->device(bridge) section name from dmmap_bridge_port->bridge_port->device_section_name
	dmmap_bridge_section = get_dup_section_in_dmmap_opt("dmmap_bridge", "device", "bridge_instance", br_inst + 1);
	dmuci_get_value_by_section_string(dmmap_bridge_section, "section_name", &br_sec_name);

	if (!dmmap_bridge_section || DM_STRLEN(br_sec_name) == 0)
		return;

	// Find the network->device(candidate bridge) section
	network_bridge_sec_from = get_origin_section_from_config("network", "device", br_sec_name);
	if (!network_bridge_sec_from)
		return;

	if (DM_LSTRCMP(component, "SVLAN") == 0) {
		struct uci_list *uci_list = NULL;
		struct uci_element *e = NULL;

		dmuci_get_value_by_section_list(network_bridge_sec_from, "ports", &uci_list);
		if (uci_list != NULL) {
			uci_foreach_element(uci_list, e) {
				// in bridge-vlan backend
				// if this port is eth2.100.300
				// then there would have been a bridge with eth2 as its port
				// so delete that bridge
				// in tradtional backend that port would have been eth2.100
				char *dev_ifname = dmstrdup(e->name);
				// see if there is a dot and replace it with null
				// so now we should have eth2 in dev_ifname
				char *dot = NULL;
				dot = DM_STRSTR(dev_ifname, ".");

				if (dot) {
					*dot = '\0';
				}

				uci_foreach_option_eq_safe("network", "device", "type", "bridge", tmp_s, s) {
					struct uci_list *br_list = NULL;
					struct uci_element *br_e = NULL;

					dmuci_get_value_by_section_list(s, "ports", &br_list);
					if (br_list == NULL)
						continue;

					uci_foreach_element(br_list, br_e) {
						if (DM_STRCMP(dev_ifname, br_e->name) == 0) {
							struct uci_section *dev_s = NULL;
							char *dev_name = NULL;

							dmuci_get_value_by_section_string(s, "name", &dev_name);

							dev_s = get_dup_section_in_config_opt("network", "interface", "device", dev_name);
							dmuci_delete_by_section(dev_s, NULL, NULL);

							dmuci_delete_by_section(s, NULL, NULL);
							break;
						}
					}
				}
			}
		}
	}

	if (network_bridge_sec_to) {
		/*
		 * The provider bridge secion has already been created(as a result of previous call to this function) in network uci file.
		 * Just need to find config section of candidate bridge and add it to the existing provider bridge configuration.
		 * And delete the candidate bridge section from network uci file.
		 *
		 */
		char *dev_name = NULL;
		dmuci_get_value_by_section_string(network_bridge_sec_from, "name", &dev_name);
		struct uci_section *dev_sec = get_dup_section_in_config_opt("network", "interface", "device", dev_name);

		char *dev_name_to = NULL;
		dmuci_get_value_by_section_string(network_bridge_sec_to, "name", &dev_name_to);

		struct uci_section *bv_to = NULL;
		bv_to = get_dup_section_in_config_opt("network", "bridge-vlan", "device", dev_name_to);

		if (DM_LSTRCMP(component, "CVLAN") == 0) {
			// network_bridge_sec_to has its own bridge-vlan section by now
			// get the port_str from vlanport device section of network_bridge_sec_from
			// or bridge-vlan section of this of network_bridge_sec_from
			// and add it to the network_bridge_sec_to's bridge vlan section
			struct uci_section *bv_from = NULL;
			bv_from = get_dup_section_in_config_opt("network", "bridge-vlan", "device", dev_name);

			// copy ports of bridge-vlan from section to bridge-vlan to section
			struct uci_list *ports_list = NULL;
			if (bv_from && bv_to) {
				dmuci_get_value_by_section_list(bv_from, "ports", &ports_list);

				if (ports_list != NULL) {
					struct uci_element *e = NULL;
					uci_foreach_element(ports_list, e) {
						dmuci_add_list_value_by_section(bv_to, "ports", e->name);
					}
				}

				dmuci_delete_by_section(bv_from, NULL, NULL);
			}

			// copy ports of bridge from to bridge to section
			dmuci_get_value_by_section_list(network_bridge_sec_from, "ports", &ports_list);
			if (ports_list != NULL) {
				struct uci_element *e = NULL;
				uci_foreach_element(ports_list, e) {
					dmuci_add_list_value_by_section(network_bridge_sec_to, "ports", e->name);
				}
			}
		} else {
			// get the list of ports for this svlan bridge
			// add them to the ports list of network_bridge_sec_to section
			// add them to the ports list of network_bridge_sec_to bridge-vlan section

			// Append ports from candidate bridge to provider bridge instance in network uci
			struct uci_list *ports_list = NULL;
			dmuci_get_value_by_section_list(network_bridge_sec_from, "ports", &ports_list);
			if (ports_list != NULL) {
				struct uci_element *e = NULL;
				uci_foreach_element(ports_list, e) {
					dmuci_add_list_value_by_section(bv_to, "ports", e->name);
					dmuci_add_list_value_by_section(network_bridge_sec_to, "ports", e->name);
				}
			}
		}

		// Delete the candidate bridge config from network uci file.
		dmuci_delete_by_section(network_bridge_sec_from, NULL, NULL);
		dmuci_delete_by_section(dev_sec, NULL, NULL);
	} else {
		/*
		 * This is the first vlan component of this provider bridge instance.
		 * Need to create a porvider bridge instance in network uci file.
		 * To create a new provider bridge instance just rename candidate bridge config section name to pr_br_{i}
		 *
		 */

		// Rename network->device(bridge) section as pr_br_{i}
		dmuci_rename_section_by_section(network_bridge_sec_from, pr_br_sec_name);

		// we need to set vlanid associated with the network_bridge_sec_from, when we will restore
		// so get the bridge-vlan section and from there the vlanid
		struct uci_section *bv = NULL;

		uci_path_foreach_option_eq(bbfdm, "dmmap_bridge_vlan", "bridge-vlan", "br_inst", br_inst + 1, bv) {
			char *vlanid = NULL;
			dmuci_get_value_by_section_string(bv, "vlan", &vlanid);

			if (DM_STRLEN(vlanid)) {
				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "vlan", vlanid);
			}

			break;
		}

		// Add option section_name to dmmap provider bridge section
		dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "section_name", pr_br_sec_name);
	}
}

static void synchronize_bridge_config_sections_with_dmmap_bridge_eq(const char *package, const char *section_type, const char *dmmap_package,
			const char *option_name, const char *option_value, struct list_head *dup_list)
{
	struct uci_section *s = NULL, *dmmap_sect = NULL;

	uci_foreach_option_eq(package, section_type, option_name, option_value, s) {

		// Skip Provider Bridge sections
		if (strncmp(section_name(s), "pr_br_", 6) == 0)
			continue;

		if ((dmmap_sect = get_dup_section_in_dmmap(dmmap_package, "device", section_name(s))) == NULL) {
			struct uci_list *ports_list = NULL;

			dmuci_add_section_bbfdm(dmmap_package, "device", &dmmap_sect);
			dmuci_set_value_by_section(dmmap_sect, "section_name", section_name(s));

			dmuci_get_value_by_section_list(s, "ports", &ports_list);
			if (ports_list != NULL) {
				struct uci_element *e = NULL;

				uci_foreach_element(ports_list, e) {
					dmuci_add_list_value_by_section(dmmap_sect, "ports", e->name);
				}
			}
		}
	}

	/*
	 * Add system and dmmap sections to the list
	 */
	uci_path_foreach_sections(bbfdm, dmmap_package, "device", dmmap_sect) {
		char *section_name = NULL;

		dmuci_get_value_by_section_string(dmmap_sect, "section_name", &section_name);
		struct uci_section *origin_s = get_origin_section_from_config(package, section_type, section_name);
		add_dmmap_config_dup_list(dup_list, origin_s, dmmap_sect);
	}
}

static bool is_bridge_section_exist(const char *device)
{
	struct uci_section *s = NULL;

	if (DM_STRLEN(device) == 0)
		return false;

	uci_path_foreach_sections(bbfdm, "dmmap_bridge", "device", s) {
		struct uci_list *ports_list = NULL;
		struct uci_element *e = NULL;

		dmuci_get_value_by_section_list(s, "ports", &ports_list);
		if (ports_list == NULL)
			continue;

		uci_foreach_element(ports_list, e) {
			if (DM_STRCMP(e->name, device) == 0)
				return true;
		}
	}

	return false;
}

static int get_last_instance_bridge(const char *package, const char *section, const char *opt_inst)
{
	struct uci_section *s;
	int inst = 0;

	uci_path_foreach_sections(bbfdm, package, section, s) {
		char *opt_val = NULL;

		dmuci_get_value_by_section_string(s, opt_inst, &opt_val);
		if (DM_STRLEN(opt_val) != 0 && DM_STRTOL(opt_val) > inst)
			inst = DM_STRTOL(opt_val);
	}

	return inst;
}

static char *create_dmmap_bridge_section(const char *port)
{
	struct uci_section *dmmap_br_sec = NULL;
	char bridge_name[64] = {0};
	char *current_inst = NULL;

	int last_inst_dmmap = get_last_instance_bridge("dmmap_bridge", "device", "bridge_instance");
	dmasprintf(&current_inst, "%d", (last_inst_dmmap == 0) ? 1 : last_inst_dmmap+1);
	snprintf(bridge_name, sizeof(bridge_name), "dev_br%d", (last_inst_dmmap == 0) ? 1 : last_inst_dmmap+1);

	dmuci_add_section_bbfdm("dmmap_bridge", "device", &dmmap_br_sec);
	dmuci_set_value_by_section(dmmap_br_sec, "section_name", bridge_name);
	dmuci_set_value_by_section(dmmap_br_sec, "bridge_instance", current_inst);
	dmuci_add_list_value_by_section(dmmap_br_sec, "ports", port);

	return current_inst;
}

static void dmmap_synchronizeBridgingProviderBridge(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct uci_section *s = NULL;

	uci_foreach_option_eq("network", "device", "type", "bridge", s) {
		struct uci_section *dmmap_pr_br_sec = NULL;
		struct uci_list *ports_list = NULL;
		struct uci_element *e = NULL;
		char current_inst[16] = {0};

		if (strncmp(section_name(s), "pr_br_", 6) != 0)
			continue;

		if ((dmmap_pr_br_sec = get_dup_section_in_dmmap("dmmap_provider_bridge", "provider_bridge", section_name(s))) != NULL)
			continue;

		int last_inst_dmmap = get_last_instance_bridge("dmmap_provider_bridge", "provider_bridge", "provider_bridge_instance");
		dmuci_add_section_bbfdm("dmmap_provider_bridge", "provider_bridge", &dmmap_pr_br_sec);
		snprintf(current_inst, sizeof(current_inst), "%d", (last_inst_dmmap == 0) ? 1 : last_inst_dmmap+1);
		dmuci_set_value_by_section(dmmap_pr_br_sec, "provider_bridge_instance", current_inst);
		dmuci_set_value_by_section(dmmap_pr_br_sec, "section_name", section_name(s));
		dmuci_set_value_by_section(dmmap_pr_br_sec, "enable", "1");

		dmuci_get_value_by_section_list(s, "ports", &ports_list);
		if (ports_list == NULL)
			continue;

		uci_foreach_element(ports_list, e) {
			struct uci_section *ss = NULL;
			bool found = false;

			uci_foreach_option_eq("network", "device", "name", e->name, ss) {
				char *type = NULL;

				found = true;
				dmuci_get_value_by_section_string(ss, "type", &type);

				// If type is 8021ad, add to svlan
				if (DM_LSTRCMP(type,"8021ad") == 0 && !is_bridge_section_exist(e->name)) {
					char *ifname = NULL;

					dmuci_get_value_by_section_string(ss, "ifname", &ifname);
					create_dmmap_bridge_section(ifname);

					// Create device bridge dmmap section
					char *svlan_br_inst = create_dmmap_bridge_section(e->name);

					// Add svlan instance to provider bridge
					dmuci_set_value_by_section(dmmap_pr_br_sec, "svlan_br_inst", svlan_br_inst);
				} else {

					// If type is bridge-vlan or transparent, add to cvlan
					if (!is_bridge_section_exist(e->name)) {
						// Create device bridge dmmap section
						char *cvlan_br_inst = create_dmmap_bridge_section(e->name);

						// Add cvlan instance to provider bridge
						dmuci_add_list_value_by_section(dmmap_pr_br_sec, "cvlan_br_inst", cvlan_br_inst);
					} else {
						// bridge section exists, add it to cvlan list
						// get bridge section
						struct uci_section *bridge_port_section = NULL;
						uci_path_foreach_option_eq(bbfdm, "dmmap_bridge_port", "bridge_port", "port", e->name,
								bridge_port_section) {
							char *br_inst = NULL;
							dmuci_get_value_by_section_string(bridge_port_section, "br_inst", &br_inst);
							if (DM_STRLEN(br_inst)) {
								dmuci_add_list_value_by_section(dmmap_pr_br_sec, "cvlan_br_inst", br_inst);
							}

							break;
						}
					}
				}
			}

			if (!found) {
				// Create device bridge dmmap section
				char *cvlan_br_inst = create_dmmap_bridge_section(e->name);

				// Add cvlan instance to provider bridge
				dmuci_add_list_value_by_section(dmmap_pr_br_sec, "cvlan_br_inst", cvlan_br_inst);
			}
		}
	}
}

static void dmmap_synchronizeBridgingBridgeVLANPort(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	// network UCI to dmmap generation would already have been done by BridgeVLAN browse function
	// what remains is to remove extra dmmap sections
	struct dm_data *args = (struct dm_data *)prev_data;
	struct uci_section *s = NULL, *stmp = NULL;

	if (!args->config_section)
		return;

	uci_path_foreach_option_eq_safe(bbfdm, "dmmap_bridge_vlanport", "bridge_vlanport", "br_inst", (char *)args->additional_data, stmp, s) {
		char *s_user = NULL;
		char *vlanport = NULL;
		char *enabled = NULL;
		char *bridge_vlan_section = NULL;
		struct uci_list *vlanports_list = NULL;

		// section added by user ==> skip it
		dmuci_get_value_by_section_string(s, "added_by_user", &s_user);
		if (DM_LSTRCMP(s_user, "1") == 0) {
			continue;
		}

		// port is disabled ==> if yes, skip it
		dmuci_get_value_by_section_string(s, "enabled", &enabled);
		if (DM_LSTRCMP(enabled, "0") == 0) {
			continue;
		}

		// get bridge vlan section name, then from section name vlanport list
		dmuci_get_value_by_section_string(s, "bridge_vlan_section", &bridge_vlan_section);

		// if no bridge vlan section set yet, then skip it
		if (DM_STRLEN(bridge_vlan_section) == 0) {
			continue;
		}

		dmuci_get_option_value_list("network", bridge_vlan_section, "ports", &vlanports_list);

		// vlanport is present in bridge-vlan => skip
		dmuci_get_value_by_section_string(s, "port", &vlanport);
		if (DM_STRLEN(vlanport) == 0) {
			continue;
		}

		if (value_exists_in_uci_list(vlanports_list, vlanport)) {
			continue;
		}

		// else ==> delete section
		dmuci_delete_by_section(s, NULL, NULL);
	}
}

static bool is_wireless_ifname_exist(const char *dev_sec_name, const char *ifname)
{
	if (DM_STRLEN(dev_sec_name) == 0 || DM_STRLEN(ifname) == 0)
		return false;

	struct uci_section *interface_s = get_dup_section_in_config_opt("network", "interface", "device", dev_sec_name);
	if (interface_s == NULL)
		return false;

	struct uci_section *s = NULL;
	uci_foreach_option_eq("wireless", "wifi-iface", "network", section_name(interface_s), s) {
		char *curr_ifname = NULL;

		/* filter out backhaul types. */
		if (multi_ap_type_backhaul(s))
			continue;

		dmuci_get_value_by_section_string(s, "ifname", &curr_ifname);
		if (DM_STRCMP(curr_ifname, ifname) == 0)
			return true;
	}

	return false;
}

static void create_new_bridge_port_section(const char *config, const char *port, const char *br_inst, const char *management_port)
{
	struct uci_section *br_port_s = NULL;

	dmuci_add_section_bbfdm("dmmap_bridge_port", "bridge_port", &br_port_s);

	dmuci_set_value_by_section(br_port_s, "config", config);
	dmuci_set_value_by_section(br_port_s, "port", port);
	dmuci_set_value_by_section(br_port_s, "br_inst", br_inst);
	dmuci_set_value_by_section(br_port_s, "management", management_port);
	dmuci_set_value_by_section(br_port_s, "enabled", "1");
}

static void dmmap_synchronizeBridgingBridgePort(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct dm_data *args = (struct dm_data *)prev_data;
	struct uci_section *s = NULL, *stmp = NULL;
	struct uci_element *e = NULL;
	char *s_user = NULL;

	if (!args->config_section)
		return;

	struct uci_list *br_ports_list = NULL;
	dmuci_get_value_by_section_list(args->config_section, "ports", &br_ports_list);

	// get name option from network/device section
	char *dev_name = NULL;
	dmuci_get_value_by_section_string(args->config_section, "name", &dev_name);

	uci_path_foreach_option_eq_safe(bbfdm, "dmmap_bridge_port", "bridge_port", "br_inst", (char *)args->additional_data, stmp, s) {

		// section added by user ==> skip it
		dmuci_get_value_by_section_string(s, "added_by_user", &s_user);
		if (DM_LSTRCMP(s_user, "1") == 0)
			continue;

		// section for management ==> skip it
		char *management = NULL;
		dmuci_get_value_by_section_string(s, "management", &management);
		if (DM_LSTRCMP(management, "1") == 0)
			continue;

		// port is disabled ==> if yes, skip it
		char *enabled = NULL;
		dmuci_get_value_by_section_string(s, "enabled", &enabled);
		if (DM_LSTRCMP(enabled, "0") == 0)
			continue;

		// port device is available in ports list ==> skip it
		char *port_device = NULL;
		dmuci_get_value_by_section_string(s, "port", &port_device);
		if (value_exists_in_uci_list(br_ports_list, port_device))
			continue;

		// check for wireless ==> skip it
		if (is_wireless_ifname_exist(dev_name, port_device))
			continue;

		// else ==> delete section
		dmuci_delete_by_section(s, NULL, NULL);
	}

	// section added by user ==> skip it
	dmuci_get_value_by_section_string(args->dmmap_section, "added_by_user", &s_user);

	if (DM_LSTRCMP(s_user, "1") != 0 && !get_section_in_dmmap_with_options_eq("dmmap_bridge_port", "bridge_port", "br_inst", (char *)args->additional_data, "management", "1"))
		create_new_bridge_port_section("network", dev_name, (char *)args->additional_data, "1");

	if (br_ports_list) {
		uci_foreach_element(br_ports_list, e) {
			if (get_section_in_dmmap_with_options_eq("dmmap_bridge_port", "bridge_port", "br_inst", (char *)args->additional_data, "port", e->name))
				continue;

			create_new_bridge_port_section("network", e->name, (char *)args->additional_data, "0");
		}
	}

	// get interface section mapped to this device name
	struct uci_section *interface_s = get_dup_section_in_config_opt("network", "interface", "device", dev_name);
	if (interface_s == NULL)
		return;

	uci_foreach_option_eq("wireless", "wifi-iface", "network", section_name(interface_s), s) {
		char *ifname = NULL;

		/* filter out backhaul types. */
		if (multi_ap_type_backhaul(s))
			continue;

		// get ifname from wireless/wifi-iface section
		dmuci_get_value_by_section_string(s, "ifname", &ifname);
		if (DM_STRLEN(ifname) == 0)
			continue;

		if (get_section_in_dmmap_with_options_eq("dmmap_bridge_port", "bridge_port", "br_inst", (char *)args->additional_data, "port", ifname))
			continue;

		create_new_bridge_port_section("wireless", ifname, (char *)args->additional_data, "0");
	}
}

static struct uci_section *get_bridge_port_device_section(struct uci_section *bridge_port_dmmap_sec)
{
	struct uci_section *s = NULL;
	char *port = NULL;

	if (!bridge_port_dmmap_sec)
		return NULL;

	/* Getting device from dmmap section */
	dmuci_get_value_by_section_string(bridge_port_dmmap_sec, "port", &port);
	if (DM_STRLEN(port) == 0)
		return NULL;

	/* Find the device port section corresponding to this device */
	if ((s = ethernet___get_ethernet_interface_section(port))) {
		return s;
	}

	/* Find the wifi-iface wireless section corresponding to this device */
	uci_foreach_option_eq("wireless", "wifi-iface", "ifname", port, s) {

		/* filter out backhaul types. */
		if (multi_ap_type_backhaul(s))
			continue;

		return s;
	}

	/* Find the device network section corresponding to this device */
	uci_foreach_option_eq("network", "device", "name", port, s) {
		return s;
	}

	return NULL;
}

static void restore_bridge_config(const char *br_inst, const char *vlanid)
{
	struct uci_section *s = NULL, *dmmap_br_sec = NULL;
	struct uci_list *br_ports_list = NULL;
	struct uci_element *e = NULL;
	char *device_section_name = NULL;
	char iface_s_name[16];
	char device_name[16];

	// Get bridge config section of vlan bridge from dmmap_bridge_port
	dmmap_br_sec = get_dup_section_in_dmmap_opt("dmmap_bridge", "device", "bridge_instance", br_inst);
	if (dmmap_br_sec == NULL)
		return;

	dmuci_get_value_by_section_string(dmmap_br_sec, "section_name", &device_section_name);
	dmuci_get_value_by_section_list(dmmap_br_sec, "ports", &br_ports_list);

	snprintf(iface_s_name, sizeof(iface_s_name), "iface_br%s", br_inst);
	snprintf(device_name, sizeof(device_name), "br-dev%s", br_inst);

	// Restore bridge config
	dmuci_add_section("network", "interface", &s);
	dmuci_rename_section_by_section(s, iface_s_name);
	dmuci_set_value_by_section(s, "device", device_name);

	dmuci_add_section("network", "device", &s);
	dmuci_rename_section_by_section(s, device_section_name);
	dmuci_set_value_by_section(s, "name", device_name);
	dmuci_set_value_by_section(s, "type", "bridge");
	dmuci_set_value_by_section(s, "bridge_empty", "1");
	if (br_ports_list) {
		uci_foreach_element(br_ports_list, e) {
			dmuci_add_list_value_by_section(s, "ports", e->name);
		}
	}

	if (DM_STRLEN(vlanid)) {
		//create bridge vlan section
		char bv_name[32] = {0};
		snprintf(bv_name, sizeof(bv_name), "br_%s_vlan_1", br_inst);

		struct uci_section *bv = NULL;
		dmuci_add_section("network", "bridge-vlan", &bv);
		dmuci_rename_section_by_section(bv, bv_name);
		dmuci_set_value_by_section(bv, "device", device_name);
		dmuci_set_value_by_section(bv, "vlan", vlanid);

		// for all vlanport sections of this bridge
		// assume one bridge had only one bridge-vlan
		// get the port_str
		// if port_str is present, add it to bridge-vlan ports list
		struct uci_section *vlanport_dmmap = NULL;

		uci_path_foreach_option_eq(bbfdm, "dmmap_bridge_vlanport", "bridge_vlanport", "br_inst", br_inst, vlanport_dmmap) {
			char *port_str = NULL;
			dmuci_get_value_by_section_string(vlanport_dmmap, "port_str", &port_str);

			if (DM_STRLEN(port_str)) {
				dmuci_add_list_value_by_section(bv, "ports", port_str);
			}
		}
	}
}

static void delete_provider_bridge(struct uci_section *data)
{
	struct uci_list *cvlan_list = NULL;
	struct uci_section *s = NULL;
	char *svlan_br_inst = NULL;
	char pr_br_inst[32] = {0};
	char *br_inst = NULL;

	/*
	 * Get cvlan/svlan bridge instance from the provider_bridge config and re-create all member bridge config section in network file.
	 * Delete all bridge_port config from dmmap_bridge_port which are member of this provider bridge.
	 * Delete provider bridge config. from network file corresponding to this provider bridge instance => config pr_br_{i}
	 * Delete this provider bridge section from dmmap_provider_bridge file.
	 *
	 */

	// Get provider bridge instance
	dmuci_get_value_by_section_string(data, "provider_bridge_instance", &br_inst);
	if (!br_inst || *br_inst == '\0')
		return;

	snprintf(pr_br_inst, sizeof(pr_br_inst), "pr_br_%s", br_inst); //name of provider bridge configuration in network file

	s = get_origin_section_from_config("network", "device", pr_br_inst);

	// Get svlan component bridge instance from dmmap section
	dmuci_get_value_by_section_string(data, "svlan_br_inst", &svlan_br_inst);

	if (svlan_br_inst && svlan_br_inst[0] != '\0') {
		// Restore bridge section in network uci file
		restore_bridge_config(svlan_br_inst, NULL);
	}

	// Get cvlan component bridge instance list from dmmap section
	dmuci_get_value_by_section_list(data, "cvlan_br_inst", &cvlan_list);

	char *vlanid = NULL;
	dmuci_get_value_by_section_string(data, "vlan", &vlanid);

	char *bridge_name = NULL;
	dmuci_get_value_by_section_string(s, "name", &bridge_name);

	if (DM_STRLEN(bridge_name)) {
		// get bridge-vlan with this name
		struct uci_section *bridge_vlan_section = NULL;
		uci_foreach_option_eq("network", "bridge-vlan", "device", bridge_name, bridge_vlan_section) {
			// delete this section and break;
			dmuci_delete_by_section(bridge_vlan_section, NULL, NULL);

			break;
		}
	}

	if (cvlan_list != NULL) {
		struct uci_element *e = NULL;

		/* traverse each list value and delete all bridge section */
		uci_foreach_element(cvlan_list, e) {
			// Restore bridge section in network uci file
			restore_bridge_config(e->name, vlanid);
		}
	}

	// Get provider bridge section from network file and delete
	dmuci_delete_by_section(s, NULL, NULL);

	// Delete dmmap bridge section.
	dmuci_delete_by_section_bbfdm(data, NULL, NULL);
}

void remove_bridge_from_provider_bridge(const char *bridge_inst)
{
	struct uci_section *pr_br_sec = NULL;

	// Traverse each provider bridge section and remove the passed bridge instance.
	// Also restore bridge in network file.
	uci_path_foreach_sections(bbfdm, "dmmap_provider_bridge", "provider_bridge", pr_br_sec) {
		struct uci_list *cvlan_list = NULL;
		char *svlan = NULL;

		// Check if the passed bridge section is svlan
		dmuci_get_value_by_section_string(pr_br_sec, "svlan_br_inst", &svlan);
		if (DM_STRCMP(svlan, bridge_inst) == 0) {
			dmuci_set_value_by_section(pr_br_sec, "svlan_br_inst", "");
		}

		// Check if the passed bridge section is cvlan
		dmuci_get_value_by_section_list(pr_br_sec, "cvlan_br_inst", &cvlan_list);
		if (cvlan_list != NULL) {
			struct uci_element *e = NULL;

			uci_foreach_element(cvlan_list, e) {
				if (DM_STRCMP(e->name, bridge_inst) == 0) {
					dmuci_del_list_value_by_section(pr_br_sec, "cvlan_br_inst", bridge_inst);
					break;
				}
			}
		}
	}
}

static void get_vlan_port_str(struct uci_section *vlanport_dmmap, char *port, char *port_str, size_t port_str_size)
{
	char *untagged = NULL;
	char *pvid = NULL;
	char *vlanid = NULL;

	if (!vlanport_dmmap || !port_str_size) {
		return;
	}

	if (!DM_STRLEN(port)) {
		dmuci_get_value_by_section_string(vlanport_dmmap, "port", &port);

		if (!DM_STRLEN(port)) {
			return;
		}
	}

	untagged = dmuci_get_value_by_section_fallback_def(vlanport_dmmap, "untagged", "1");

	dmuci_get_value_by_section_string(vlanport_dmmap, "pvid", &pvid);
	dmuci_get_value_by_section_string(vlanport_dmmap, "vlan", &vlanid);
	// if pvid is set and it is not equal to 1
	// if vlanid is set and pvid is eqaul to vlanid
	if (DM_STRLEN(pvid) && DM_STRLEN(vlanid) && DM_STRCMP(pvid, "1") && !DM_STRCMP(pvid, vlanid)) {
		snprintf(port_str, port_str_size, "%s:u*", port);
	} else {
		// check if untagged is set and if it is 1
		if (DM_STRLEN(untagged) && !DM_STRCMP(untagged, "1")) {
			snprintf(port_str, port_str_size, "%s:u", port);
		} else {
			snprintf(port_str, port_str_size, "%s:t", port);
		}
	}
}

static void create_dmmap_vlanport_section(const char *port, const char *br_inst, const char *bridge_vlan_section, const char *vlan_inst, const char *vid)
{
	if (!DM_STRLEN(br_inst) || !DM_STRLEN(port) || !DM_STRLEN(bridge_vlan_section) || !DM_STRLEN(vlan_inst))
		return;

	struct uci_section *vlanport_section = NULL;
	char *bridge_port_instance = NULL;
	struct uci_section *bridge_port_section = NULL;

	// remove the colon if present
	char *ifname = dmstrdup(port);
	char *colon = NULL;

	colon = DM_STRCHR(ifname, ':');
	if (colon)
		*colon = '\0';

	bridge_port_section = get_section_in_dmmap_with_options_eq("dmmap_bridge_port", "bridge_port", "br_inst", br_inst, "port", ifname);
	if (!bridge_port_section)
		return;

	dmuci_get_value_by_section_string(bridge_port_section, "bridge_port_instance", &bridge_port_instance);

	if (!DM_STRLEN(bridge_port_instance))
		return;

	dmuci_add_section_bbfdm("dmmap_bridge_vlanport", "bridge_vlanport", &vlanport_section);

	dmuci_set_value_by_section(vlanport_section, "port_str", port);
	dmuci_set_value_by_section(vlanport_section, "br_inst", br_inst);
	dmuci_set_value_by_section(vlanport_section, "bridge_vlan_section", bridge_vlan_section);
	dmuci_set_value_by_section(vlanport_section, "bridge_vlan_instance", vlan_inst);
	dmuci_set_value_by_section(vlanport_section, "bridge_port_instance", bridge_port_instance);
	dmuci_set_value_by_section(vlanport_section, "vlan", vid);
	dmuci_set_value_by_section(vlanport_section, "enabled", "1");
}

static void add_vlanport_sections_from_bridge_vlan(struct dm_data *curr_data, const char *vlan_inst)
{
	if (!curr_data || !curr_data->config_section || !DM_STRLEN(vlan_inst)) {
		return;
	}

	struct uci_list *ports_list = NULL;
	struct uci_element *e = NULL;
	char *br_inst = (char *)curr_data->additional_data;
	struct uci_section *bridge_vlan_section = curr_data->config_section;
	char *vid = NULL;

	// get ports list, may or may not be present
	dmuci_get_value_by_section_list(bridge_vlan_section, "ports", &ports_list);

	// get vid, may or may not be present
	dmuci_get_value_by_section_string(bridge_vlan_section, "vlan", &vid);

	if (ports_list) {
		// for each port, create vlanport section
		uci_foreach_element(ports_list, e) {
			// if section already present, nothing to do
			if (get_section_in_dmmap_with_options_eq("dmmap_bridge_vlanport", "bridge_vlanport", "br_inst", br_inst, "port_str", e->name)) {
				continue;
			}

			// section not present, add it
			create_dmmap_vlanport_section(e->name, br_inst, section_name(bridge_vlan_section),
					vlan_inst, vid);
		}
	}
}

static void add_port_to_vlans(struct dm_data *data, char *port, const char *instance, bool port_enabled, bool bridge_port)
{
	struct uci_section *vlanport_dmmap = NULL;

	if (!data || !DM_STRLEN(port) || !DM_STRLEN(instance)) {
		return;
	}

	struct dm_data *bridge_data = ((struct dm_data *)data)->additional_data;
	char *br_inst = (char *)(bridge_data->additional_data);

	// we cannot loop over bridge-vlan sections because that does not tell us whether
	// vlanport is present or not
	// we will have to loop over vlanport dmmap sections
	// for all vlanport sections with
	// if br_inst == this bridge inst
	// and if br_port_inst == this br_port_inst
	// if they have a bridge_vlan_instance set
	// if bridge_port is set
	// then create an 8021ad section based on bridge_vlan
	// else use the port from get_vlan_port_str
	// then the port is also added to the config
	// the port is set in dmmap section anyway
	uci_path_foreach_option_eq(bbfdm, "dmmap_bridge_vlanport", "bridge_vlanport", "br_inst", br_inst, vlanport_dmmap) {
		// get bridge_port_instance
		char *bridge_port_instance = NULL;
		dmuci_get_value_by_section_string(vlanport_dmmap, "bridge_port_instance", &bridge_port_instance);

		if (!DM_STRCMP(bridge_port_instance, instance)) {
			if (bridge_port) {
				// get the vid
				char *vid = NULL;
				dmuci_get_value_by_section_string(vlanport_dmmap, "vlan", &vid);

				char *vlanport_enabled = dmuci_get_value_by_section_fallback_def(vlanport_dmmap, "enabled", "1");
				if (DM_STRLEN(vid) && !DM_STRCMP(vlanport_enabled, "1") && port_enabled && bridge_data->config_section) {
					// create 8021ad section based on this vid
					struct uci_section *qinq_section = NULL;
					dmuci_add_section("network", "device", &qinq_section);
					dmuci_set_value_by_section(qinq_section, "type", "8021ad");
					dmuci_set_value_by_section(qinq_section, "vid", vid);
					dmuci_set_value_by_section(qinq_section, "ifname", port);

					char qinq_name[32] = {0};
					snprintf(qinq_name, sizeof(qinq_name), "%s.%s", port, vid);
					dmuci_set_value_by_section(qinq_section, "name", qinq_name);

					// update bridge config section
					dmuci_del_list_value_by_section(bridge_data->config_section, "ports", port);
					dmuci_del_list_value_by_section(bridge_data->dmmap_section, "ports", port);

					dmuci_add_list_value_by_section(bridge_data->config_section, "ports", qinq_name);
					dmuci_add_list_value_by_section(bridge_data->dmmap_section, "ports", qinq_name);

					// update vlanport port_str
					dmuci_set_value_by_section(vlanport_dmmap, "port_str", qinq_name);
					// also set in bridge port dmmap section
					if (data->dmmap_section) {
						dmuci_set_value_by_section(data->dmmap_section, "port", qinq_name);
					}
				}

				dmuci_set_value_by_section(vlanport_dmmap, "8021q", "1");
				// do not set anything in bridge_vlan_section
				// in fact delete bridge_vlan_section
				char *bridge_vlan_name = NULL;
				dmuci_get_value_by_section_string(vlanport_dmmap, "bridge_vlan_section", &bridge_vlan_name);

				if (DM_STRLEN(bridge_vlan_name)) {
					struct uci_section *bridge_vlan_config_s = NULL;
					bridge_vlan_config_s = get_origin_section_from_config("network", "bridge-vlan", bridge_vlan_name);
					if (bridge_vlan_config_s) {
						dmuci_delete_by_section(bridge_vlan_config_s, NULL, NULL);
					}

					struct uci_section *bridge_vlan_dmmap_s = NULL;
					bridge_vlan_dmmap_s = get_dup_section_in_dmmap_opt("dmmap_bridge_vlan", "bridge_vlan", "section_name", bridge_vlan_name);
					if (bridge_vlan_dmmap_s) {
						dmuci_delete_by_section(bridge_vlan_dmmap_s, NULL, NULL);
					}
				}
			} else {
				// if bridge_vlan_section is set then set it there too
				char *bridge_vlan_section = NULL;
				dmuci_get_value_by_section_string(vlanport_dmmap, "bridge_vlan_section", &bridge_vlan_section);

				char port_str[32] = {0};

				// generate port str based on pvid, tagged, untagged
				get_vlan_port_str(vlanport_dmmap, port, port_str, sizeof(port_str));

				// set it in network config if vlanport is enabled and bridge_vlan_section is present
				char *vlanport_enabled = dmuci_get_value_by_section_fallback_def(vlanport_dmmap, "enabled", "1");

				if (DM_STRLEN(bridge_vlan_section) && !DM_STRCMP(vlanport_enabled, "1") && port_enabled) {
					dmuci_add_list_value("network", bridge_vlan_section, "ports", port_str);
					// set it in dmmap also
					dmuci_set_value_by_section(vlanport_dmmap, "port_str", port_str);
				}
			}

			dmuci_set_value_by_section(vlanport_dmmap, "port", port);
		}
	}
}

static void remove_port_from_vlans(const char *br_inst, struct uci_section *bridge_section, const char *port, bool remove_inst)
{
	char *bridge_name = NULL;
	struct uci_section *bridge_vlan_section = NULL;
	struct uci_section *vlanport_dmmap_section = NULL;

	if (!DM_STRLEN(br_inst) || !bridge_section || !DM_STRLEN(port)) {
		return;
	}

	dmuci_get_value_by_section_string(bridge_section, "name", &bridge_name);

	if (!DM_STRLEN(bridge_name)) {
		return;
	}

	// remove from bridge-vlan sections
	uci_foreach_option_eq("network", "bridge-vlan", "device", bridge_name, bridge_vlan_section) {
		struct uci_list *ports_list = NULL;
		struct uci_element *e = NULL;

		dmuci_get_value_by_section_list(bridge_vlan_section, "ports", &ports_list);
		if (!ports_list)
			continue;

		uci_foreach_element(ports_list, e) {
			if (DM_STRSTR(e->name, port)) {
				dmuci_del_list_value_by_section(bridge_vlan_section, "ports", e->name);
			}
		}
	}

	// sections that are present in dmmap but not have been enabled yet
	// synchronize will skip them if option port is not set, so remove option port
	// for all vlanport sections of this bridge
	// if port equal this disabled port
	// then remove port
	uci_path_foreach_option_eq(bbfdm, "dmmap_bridge_vlanport", "bridge_vlanport", "br_inst", br_inst, vlanport_dmmap_section) {
		char *vlanport = NULL;
		dmuci_get_value_by_section_string(vlanport_dmmap_section, "port", &vlanport);

		if (DM_STRSTR(port, vlanport)) {
			dmuci_set_value_by_section(vlanport_dmmap_section, "port", "");
			dmuci_set_value_by_section(vlanport_dmmap_section, "port_str", "");

			// the port is being deleted, remove bridge_port_instance so nothing is displayed in
			// get function
			if (remove_inst) {
				dmuci_set_value_by_section(vlanport_dmmap_section, "bridge_port_instance", "");
			}
		}
	}
}

static void create_8021ad_device_section(char *dest_str, size_t dest_str_size, const char *port, const char *vlan,
		const char *br_inst, const char *vlanport_inst, struct uci_section *vlanport_dmmap_section)
{
	if (!dest_str_size || !DM_STRLEN(port) || !DM_STRLEN(vlan) || !DM_STRLEN(br_inst) || !DM_STRLEN(vlanport_inst)) {
		return;
	}

	snprintf(dest_str, dest_str_size, "%s.%s", port, vlan);

	char qinq_section_name[32] = {0};
	snprintf(qinq_section_name, sizeof(qinq_section_name), "br_%s_dev_%s", br_inst, vlanport_inst);

	struct uci_section *qinq_section = NULL;
	dmuci_add_section("network", "device", &qinq_section);
	dmuci_rename_section_by_section(qinq_section, qinq_section_name);
	dmuci_set_value_by_section(qinq_section, "type", "8021ad");
	dmuci_set_value_by_section(qinq_section, "ifname", port);
	dmuci_set_value_by_section(qinq_section, "name", dest_str);
	dmuci_set_value_by_section(qinq_section, "vid", vlan);

	dmuci_set_value_by_section(vlanport_dmmap_section, "qinq_section_name", qinq_section_name);
}

/*************************************************************
* ENTRY METHOD
**************************************************************/
/*#Device.Bridging.Bridge.{i}.!UCI:network/device/dmmap_bridge*/
static int browseBridgingBridgeInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct dm_data *curr_data = NULL;
	char *inst = NULL;
	LIST_HEAD(dup_list);

	synchronize_bridge_config_sections_with_dmmap_bridge_eq("network", "device", "dmmap_bridge", "type", "bridge", &dup_list);
	list_for_each_entry(curr_data, &dup_list, list) {

		inst = handle_instance(dmctx, parent_node, curr_data->dmmap_section, "bridge_instance", "bridge_alias");

		if (curr_data->config_section == NULL)
			curr_data->config_section = curr_data->dmmap_section;

		curr_data->additional_data = (void *)inst;

		if (DM_LINK_INST_OBJ(dmctx, parent_node, curr_data, inst) == DM_STOP)
			break;
	}
	free_dmmap_config_dup_list(&dup_list);
	return 0;
}

static int browseBridgingProviderBridgeInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct dm_data curr_data = {0};
	struct uci_section *s = NULL;
	char *inst = NULL;

	dmmap_synchronizeBridgingProviderBridge(dmctx, parent_node, prev_data, prev_instance);
	uci_path_foreach_sections(bbfdm, "dmmap_provider_bridge", "provider_bridge", s) {

		curr_data.config_section = s;

		inst = handle_instance(dmctx, parent_node, s, "provider_bridge_instance", "provider_bridge_alias");

		if (DM_LINK_INST_OBJ(dmctx, parent_node, &curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseBridgingBridgePortInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct dm_data curr_data = {0};
	struct uci_section *dmmap_s = NULL;
	char *inst = NULL;

	dmmap_synchronizeBridgingBridgePort(dmctx, parent_node, prev_data, prev_instance);
	uci_path_foreach_option_eq(bbfdm, "dmmap_bridge_port", "bridge_port", "br_inst", (char *)((struct dm_data *)prev_data)->additional_data, dmmap_s) {

		curr_data.config_section = get_bridge_port_device_section(dmmap_s);
		curr_data.dmmap_section = dmmap_s;
		curr_data.additional_data = prev_data;

		inst = handle_instance(dmctx, parent_node, dmmap_s, "bridge_port_instance", "bridge_port_alias");

		if (DM_LINK_INST_OBJ(dmctx, parent_node, &curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

// delete bridge-vlans that do not have a bridge
// if bridge is present, make sure br_inst is updated in bridge-vlan dmmap section
static void update_bridge_vlan_dmmap(struct dm_data *curr_data)
{
	// check if br_inst is set or not
	char *br_inst = NULL;
	dmuci_get_value_by_section_string(curr_data->dmmap_section, "br_inst", &br_inst);

	if (!DM_STRLEN(br_inst)) {
		struct uci_section *bridge_sec = NULL;
		char *bridge = NULL;
		dmuci_get_value_by_section_string(curr_data->config_section, "device", &bridge);

		if (!DM_STRLEN(bridge)) {
			dmuci_delete_by_section(curr_data->dmmap_section, NULL, NULL);
			dmuci_delete_by_section(curr_data->config_section, NULL, NULL);
		} else {
			bridge_sec = get_dup_section_in_config_opt("network", "device", "name", bridge);

			if (!bridge_sec) {
				dmuci_delete_by_section(curr_data->dmmap_section, NULL, NULL);
				dmuci_delete_by_section(curr_data->config_section, NULL, NULL);
				
			} else {
				struct uci_section *dmmap_bridge = NULL;
				dmmap_bridge = get_dup_section_in_dmmap_opt("dmmap_bridge", "device", "section_name",
										section_name(bridge_sec));

				if (dmmap_bridge) {
					br_inst = NULL;

					dmuci_get_value_by_section_string(dmmap_bridge, "bridge_instance", &br_inst);
					dmuci_set_value_by_section(curr_data->dmmap_section, "br_inst", br_inst);
				}
			}
		}
	}
}

static int browseBridgingBridgeVLANInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	char *br_inst = (char *)((struct dm_data *)prev_data)->additional_data;
	char *inst = NULL;
	struct dm_data *curr_data = NULL;
	LIST_HEAD(dup_list);

	synchronize_specific_config_sections_with_dmmap("network", "bridge-vlan", "dmmap_bridge_vlan", &dup_list);
	list_for_each_entry(curr_data, &dup_list, list) {
		// save br_inst
		curr_data->additional_data = prev_data;
		char *bridge_vlan_bridge_instance = NULL;

		// some options might need to be set manually in bridge_vlan_dmmap
		update_bridge_vlan_dmmap(curr_data);

		// if bridge instances do not match, skip
		dmuci_get_value_by_section_string(curr_data->dmmap_section, "br_inst", &bridge_vlan_bridge_instance);
		if (DM_STRCMP(br_inst, bridge_vlan_bridge_instance)) {
			continue;
		}

		inst = handle_instance(dmctx, parent_node, curr_data->dmmap_section, "bridge_vlan_instance", "bridge_vlan_alias");

		// since each port present in this bridge-vlan section corresponds to VLANPort object
		// add dmmap_sections for those too
		add_vlanport_sections_from_bridge_vlan(curr_data, inst);

		char *vlan = NULL;
		// for convenience
		dmuci_get_value_by_section_string(curr_data->config_section, "vlan", &vlan);
		if (DM_STRLEN(vlan))
			dmuci_set_value_by_section(curr_data->dmmap_section, "vlan", vlan);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)curr_data, inst) == DM_STOP)
			break;
	}

	free_dmmap_config_dup_list(&dup_list);
	return 0;
}

static int browseBridgingBridgeVLANPortInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	char *br_inst = (char *)((struct dm_data *)prev_data)->additional_data;
	struct uci_section *br_vlanport_dmmap_s = NULL;
	char *inst = NULL;

	dmmap_synchronizeBridgingBridgeVLANPort(dmctx, parent_node, prev_data, prev_instance);
	uci_path_foreach_option_eq(bbfdm, "dmmap_bridge_vlanport", "bridge_vlanport", "br_inst", br_inst, br_vlanport_dmmap_s) {
		struct dm_data curr_data = {0};

		curr_data.dmmap_section = br_vlanport_dmmap_s;

		//curr_data.additional_data = (void *)(br_inst);
		curr_data.additional_data = prev_data;

		inst = handle_instance(dmctx, parent_node, br_vlanport_dmmap_s, "bridge_vlanport_instance", "bridge_vlanport_alias");

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

/*************************************************************
* ADD DELETE OBJECT
**************************************************************/
static int addObjBridgingBridge(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct uci_section *s = NULL, *dmmap_bridge = NULL;
	char iface_s_name[16];
	char dev_s_name[16];
	char device_name[16];

	snprintf(iface_s_name, sizeof(iface_s_name), "iface_br%s", *instance);
	snprintf(dev_s_name, sizeof(dev_s_name), "dev_br%s", *instance);
	snprintf(device_name, sizeof(device_name), "br-dev%s", *instance);

	// Add network bridge section
	dmuci_add_section("network", "interface", &s);
	dmuci_rename_section_by_section(s, iface_s_name);
	dmuci_set_value_by_section(s, "device", device_name);

	// Add device bridge section
	dmuci_add_section("network", "device", &s);
	dmuci_rename_section_by_section(s, dev_s_name);
	dmuci_set_value_by_section(s, "name", device_name);
	dmuci_set_value_by_section(s, "type", "bridge");
	dmuci_set_value_by_section(s, "bridge_empty", "1");

	// Add dmmap bridge section
	dmuci_add_section_bbfdm("dmmap_bridge", "device", &dmmap_bridge);
	dmuci_set_value_by_section(dmmap_bridge, "section_name", section_name(s));
	dmuci_set_value_by_section(dmmap_bridge, "added_by_user", "1");
	dmuci_set_value_by_section(dmmap_bridge, "bridge_instance", *instance);
	return 0;
}

static int delObjBridgingBridge(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	struct uci_section *s = NULL, *stmp = NULL;

	switch (del_action) {
		case DEL_INST:
			// Remove all bridge port sections related to this device bridge section
			uci_path_foreach_option_eq_safe(bbfdm, "dmmap_bridge_port", "bridge_port", "br_inst", (char *)((struct dm_data *)data)->additional_data, stmp, s) {
				dmuci_delete_by_section(s, NULL, NULL);
			}

			// Remove all bridge vlan sections related to this device bridge section
			uci_path_foreach_option_eq_safe(bbfdm, "dmmap_bridge_vlan", "bridge-vlan", "br_inst", (char *)((struct dm_data *)data)->additional_data, stmp, s) {
				char *sec_name = NULL;

				dmuci_get_value_by_section_string(s, "section_name", &sec_name);
				if (DM_STRLEN(sec_name)) {
					struct uci_section *device_s = get_origin_section_from_config("network", "bridge-vlan", sec_name);
					dmuci_delete_by_section(device_s, NULL, NULL);
				}

				dmuci_delete_by_section(s, NULL, NULL);
			}

			// Remove all bridge vlanport sections related to this device bridge section
			uci_path_foreach_option_eq_safe(bbfdm, "dmmap_bridge_vlanport", "bridge_vlanport", "br_inst", (char *)((struct dm_data *)data)->additional_data, stmp, s) {
				// maybe this was a pb scenario and an 8021q or 8021ad device section is present, delete it
				char *port = NULL, *port_str = NULL, *dot = NULL;
				dmuci_get_value_by_section_string(s, "port", &port);

				if (DM_STRLEN(port)) {
					dot = DM_STRCHR(port, '.');
					if (dot) {
						struct uci_section *vlandev = NULL;
						vlandev = get_dup_section_in_config_opt("network", "device", "name", port);

						if (vlandev) {
							dmuci_delete_by_section(vlandev, NULL, NULL);
						}
					}
				}

				port_str = NULL;
				dmuci_get_value_by_section_string(s, "port_str", &port_str);
				if (DM_STRLEN(port_str)) {
					dot = DM_STRCHR(port_str, '.');
					if (dot) {
						struct uci_section *vlandev = NULL;
						vlandev = get_dup_section_in_config_opt("network", "device", "name", port_str);

						if (vlandev) {
							dmuci_delete_by_section(vlandev, NULL, NULL);
						}
					}
				}

				dmuci_delete_by_section(s, NULL, NULL);
			}

			// Remove cvlan/svaln from dmmap_provider_bridge section if this bridge instance is a part of it
			remove_bridge_from_provider_bridge((char *)((struct dm_data *)data)->additional_data);

			// Remove interface bridge that maps to this device
			remove_device_from_bridge_interface(((struct dm_data *)data)->config_section);

			struct uci_section *bridge_dmmap_section = NULL;
			bridge_dmmap_section = get_dup_section_in_dmmap("dmmap_bridge", "device", section_name(((struct dm_data *)data)->config_section));

			if (bridge_dmmap_section) {
				dmuci_delete_by_section_bbfdm(bridge_dmmap_section, NULL, NULL);
			}

			// Remove device bridge section
			if (((struct dm_data *)data)->config_section) {
				dmuci_delete_by_section(((struct dm_data *)data)->config_section, NULL, NULL);
			}

			break;
		case DEL_ALL:
			break;
	}
	return 0;
}

static int addObjBridgingBridgePort(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct uci_section *br_port_s = NULL;
	char buf[32];

	snprintf(buf, sizeof(buf), "br_%s_port_%s", (char *)((struct dm_data *)data)->additional_data, *instance);

	// Add dmmap section for devices
	dmuci_add_section_bbfdm("dmmap_bridge_port", "bridge_port", &br_port_s);
	dmuci_rename_section_by_section(br_port_s, buf);

	dmuci_set_value_by_section(br_port_s, "config", "network");
	dmuci_set_value_by_section(br_port_s, "br_inst", (char *)((struct dm_data *)data)->additional_data);
	dmuci_set_value_by_section(br_port_s, "bridge_port_instance", *instance);
	dmuci_set_value_by_section(br_port_s, "management", "0");
	dmuci_set_value_by_section(br_port_s, "enabled", "0");
	dmuci_set_value_by_section(br_port_s, "added_by_user", "1");
	return 0;
}

static int delObjBridgingBridgePort(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	switch (del_action) {
	case DEL_INST:
		if (!bridging___is_management_port_instance(((struct dm_data *)data)->dmmap_section)) {
			struct dm_data *bridge_args = ((struct dm_data *)data)->additional_data;
			char *enable = NULL, *port = NULL;

			// Get enabled and port options
			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "enabled", &enable);
			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "port", &port);

			if (DM_STRCMP(enable, "1") == 0) {
				// Remove port from port list interface
				remove_port_from_bridge_sections(bridge_args->config_section, bridge_args->dmmap_section, port);
			}

			char *device = NULL;
			dmuci_get_value_by_section_string(bridge_args->config_section, "name", &device);
			// the last argument is the flag that indicates this is a deletion
			remove_port_from_vlans((char *)(bridge_args->additional_data), bridge_args->config_section, port, 1);
		}

		// Remove dmmap section
		dmuci_delete_by_section_bbfdm(((struct dm_data *)data)->dmmap_section, NULL, NULL);
		break;
	case DEL_ALL:
		break;
	}
	return 0;
}

static int addObjBridgingBridgeVLANPort(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct uci_section *br_vlanport_s = NULL;
	char device_name[32];

	snprintf(device_name, sizeof(device_name), "br_%s_vlanport_%s", (char *)((struct dm_data *)data)->additional_data, *instance);

	// Add dmmap section
	dmuci_add_section_bbfdm("dmmap_bridge_vlanport", "bridge_vlanport", &br_vlanport_s);
	dmuci_set_value_by_section(br_vlanport_s, "br_inst", (char *)((struct dm_data *)data)->additional_data);
	dmuci_set_value_by_section(br_vlanport_s, "bridge_vlanport_instance", *instance);
	dmuci_set_value_by_section(br_vlanport_s, "added_by_user", "1");
	dmuci_set_value_by_section(br_vlanport_s, "enabled", "0");

	return 0;
}

static int delObjBridgingBridgeVLANPort(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	char *port = NULL;
	char *bridge_vlan_section = NULL;
	struct uci_section *vlanport_section = ((struct dm_data *)data)->dmmap_section;

	// get the bridge-vlan section for this vlanport
	// if not set, just remove this dmmap section
	// if set, remove the port from the list and remove this dmmap section
	switch (del_action) {
		case DEL_INST:
			dmuci_get_value_by_section_string(vlanport_section, "port", &port);
			dmuci_get_value_by_section_string(vlanport_section, "bridge_vlan_section", &bridge_vlan_section);

			if (DM_STRLEN(port) && DM_STRLEN(bridge_vlan_section)) {
				char port_str[32] = {0};
				get_vlan_port_str(vlanport_section, port, port_str, sizeof(port_str));
				// delete from list
				dmuci_del_list_value("network", bridge_vlan_section, "ports", port_str);
			}

			// Remove dmmap section
			dmuci_delete_by_section_bbfdm(vlanport_section, NULL, NULL);
			break;
		case DEL_ALL:
			break;
	}

	return 0;
}

static int addObjBridgingBridgeVLAN(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct dm_data *bridge_args = (struct dm_data *)data;
	struct uci_section *br_vlan_s = NULL;
	char br_vlan_name[32]= {0};
	char *br_name = NULL;
	struct uci_section *s = NULL;

	// get the name of the bridge
	dmuci_get_value_by_section_string(bridge_args->config_section, "name", &br_name);
	if (!DM_STRLEN(br_name)) {
		return 1;
	}

	// starter vlan name
	snprintf(br_vlan_name, sizeof(br_vlan_name), "br_%s_vlan_%s", (char *)(bridge_args->additional_data), *instance);

	// add bridge-vlan section in network uci
	dmuci_add_section("network", "bridge-vlan", &s);
	dmuci_rename_section_by_section(s, br_vlan_name);
	dmuci_set_value_by_section(s, "device", br_name);

	// add bbf section for bridge vlan
	dmuci_add_section_bbfdm("dmmap_bridge_vlan", "bridge-vlan", &br_vlan_s);
	dmuci_set_value_by_section(br_vlan_s, "section_name", br_vlan_name);
	dmuci_set_value_by_section(br_vlan_s, "br_inst", (char *)(bridge_args->additional_data));
	dmuci_set_value_by_section(br_vlan_s, "bridge_vlan_instance", *instance);

	return 0;
}

static void remove_vlan_from_vlan_port_sections(struct dm_data *data, char *instance)
{
	if (!data || !DM_STRLEN(instance))
		return;

	struct uci_section *vlanport_dmmap_section = NULL;
	char *br_inst = (char *)(data->additional_data);

	// loop over all VLAN Port sections for this bridge
	uci_path_foreach_option_eq(bbfdm, "dmmap_bridge_vlanport", "bridge_vlanport", "br_inst", br_inst, vlanport_dmmap_section) {
		char *bridge_vlan_instance = NULL;
		dmuci_get_value_by_section_string(vlanport_dmmap_section, "bridge_vlan_instance", &bridge_vlan_instance);
		// if VLANPort belongs to same section as this bridge-vlan section
		if (!DM_STRCMP(bridge_vlan_instance, instance)) {
			dmuci_set_value_by_section(vlanport_dmmap_section, "bridge_vlan_section", "");
			dmuci_set_value_by_section(vlanport_dmmap_section, "bridge_vlan_instance", "");
		}
	}
}

static int addObjBridgingProviderBridge(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct uci_section *pr_br_sec = NULL;

	// Add dmmap section
	dmuci_add_section_bbfdm("dmmap_provider_bridge", "provider_bridge", &pr_br_sec);
	dmuci_set_value_by_section(pr_br_sec, "enable", "1");
	dmuci_set_value_by_section(pr_br_sec, "provider_bridge_instance", *instance);
	return 0;
}

static int delObjBridgingProviderBridge(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	switch (del_action) {
	case DEL_INST:
		delete_provider_bridge(((struct dm_data *)data)->config_section);
		break;
	case DEL_ALL:
		break;
	}
	return 0;
}

static int delObjBridgingBridgeVLAN(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	switch (del_action) {
		case DEL_INST:
			// unset vlan_inst, vlan_id and bridge_vlan_section from VP dmmap sections
			remove_vlan_from_vlan_port_sections((struct dm_data *)data, instance);
			// delete bbfdm section
			dmuci_delete_by_section(((struct dm_data *)data)->dmmap_section, NULL, NULL);
			// delete network uci section
			dmuci_delete_by_section(((struct dm_data *)data)->config_section, NULL, NULL);
			break;
		case DEL_ALL:
			break;
	}
	return 0;
}

/**************************************************************************
*SET & GET PARAMETERS
***************************************************************************/
static int get_Bridging_MaxBridgeEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("20");
	return 0;
}

static int get_Bridging_MaxDBridgeEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("20");
	return 0;
}

static int get_Bridging_MaxQBridgeEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("20");
	return 0;
}

static int get_Bridging_MaxVLANEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("20");
	return 0;
}

static int get_Bridging_MaxProviderBridgeEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("20");
	return 0;
}

static int get_Bridging_get_Bridging_MaxFilterEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("0");
	return 0;
}

/*#Device.Bridging.ProviderBridgeNumberOfEntries!UCI:network/device/*/
static int get_Bridging_ProviderBridgeNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseBridgingProviderBridgeInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

/*#Device.Bridging.BridgeNumberOfEntries!UCI:network/device/*/
static int get_Bridging_BridgeNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseBridgingBridgeInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

/*#Device.Bridging.Bridge.{i}.Enable!UCI:network/device,@i-1/enabled*/
static int get_BridgingBridge_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "enabled", "1");
	return 0;
}

static int set_BridgingBridge_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "enabled", b ? "1" : "0");
			return 0;
	}
	return 0;
}

/*#Device.Bridging.Bridge.{i}.Status!UCI:network/device,@i-1/enabled*/
static int get_BridgingBridge_Status(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *name = NULL;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "name", &name);
	get_net_device_status(name, value);
	if (DM_STRCMP(*value, "Up") == 0) {
		*value = dmstrdup("Enabled");
	} else {
		*value = dmstrdup("Disabled");
	}
	return 0;
}

static int get_BridgingBridge_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return bbf_get_alias(ctx, ((struct dm_data *)data)->dmmap_section, "bridge_alias", instance, value);
}

static int set_BridgingBridge_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	return bbf_set_alias(ctx, ((struct dm_data *)data)->dmmap_section, "bridge_alias", instance, value);
}

static int get_BridgingBridge_Name(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmasprintf(value, "bridge-%s", (char *)((struct dm_data *)data)->additional_data);
	return 0;
}

static int get_BridgingBridge_Standard(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("802.1Q-2011");
	return 0;
}

static int set_BridgingBridge_Standard(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *Bridge_Standard[] = {"802.1D-2004", "802.1Q-2005", "802.1Q-2011", NULL};

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, Bridge_Standard, NULL))
				return FAULT_9007;
			return 0;
		case VALUESET:
			return 0;
	}
	return 0;
}

static int get_BridgingBridge_PortNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseBridgingBridgePortInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_BridgingBridge_VLANNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseBridgingBridgeVLANInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_BridgingBridge_VLANPortNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseBridgingBridgeVLANPortInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}


static int get_BridgingBridgeSTP_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "stp", "0");
	return 0;
}

static int set_BridgingBridgeSTP_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "stp", b ? "1" : "0");
			return 0;
	}
	return 0;
}

static int get_BridgingBridgeSTP_Status(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *name = NULL;

	*value = dmstrdup("Disabled");
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "name", &name);
	if (DM_STRLEN(name) == 0) {
		return 0;
	}

	char *enable = NULL;
	get_net_device_sysfs(name, "bridge/stp_state", &enable);
	if (DM_STRCMP(enable, "1") == 0)
		*value = dmstrdup("Enabled");

	return 0;
}

static int get_BridgingBridgeSTP_Protocol(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("STP");
	return 0;
}

static int set_BridgingBridgeSTP_Protocol(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *Protocol[] = {"STP", NULL};

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, Protocol, NULL))
				return FAULT_9007;
			return 0;
		case VALUESET:
			return 0;
	}
	return 0;
}

static int get_BridgingBridgeSTP_BridgePriority(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "priority", "32767");
	return 0;
}

static int set_BridgingBridgeSTP_BridgePriority(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"0","61440"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "priority", value);
			return 0;
	}
	return 0;
}

static int get_BridgingBridgeSTP_HelloTime(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "hello_time", value);

	// Value defined in system is in seconds but in datamodel this is in centiseconds, convert the value to centiseconds
	int hello_time = DM_STRLEN(*value) ? DM_STRTOL(*value) * 100 : 200;

	dmasprintf(value, "%d", hello_time);
	return 0;
}

static int set_BridgingBridgeSTP_HelloTime(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char buf[16] = {0};

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"100","1000"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			// Value defined in system is in seconds but in datamodel this is in centiseconds, convert the value to seconds
			snprintf(buf, sizeof(buf), "%u", (uint32_t)DM_STRTOL(value) / 100);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "hello_time", buf);
			return 0;
	}
	return 0;
}

static int get_BridgingBridgeSTP_MaxAge(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "max_age", value);

	// Value defined in system is in seconds but in datamodel this is in centiseconds, convert the value to centiseconds
	int max_age = DM_STRLEN(*value) ? DM_STRTOL(*value) * 100 : 2000;

	dmasprintf(value, "%d", max_age);
	return 0;
}

static int set_BridgingBridgeSTP_MaxAge(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char buf[16] = {0};

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"600","4000"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			// Value defined in system is in seconds but in datamodel this is in centiseconds, convert the value to seconds
			snprintf(buf, sizeof(buf), "%u", (uint32_t)DM_STRTOL(value) / 100);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "max_age", buf);
			return 0;
	}
	return 0;
}

static int get_BridgingBridgeSTP_ForwardingDelay(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "forward_delay", "4");
	return 0;
}

static int set_BridgingBridgeSTP_ForwardingDelay(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"4","30"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "forward_delay", value);
			return 0;
	}
	return 0;
}

static int get_BridgingBridgePort_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "enabled", value);
	return 0;
}

static int set_BridgingBridgePort_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct dm_data *bridge_args = ((struct dm_data *)data)->additional_data;
	char *config = NULL, *device = NULL;
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;

			return 0;
		case VALUESET:
			string_to_bool(value, &b);

			if (bridging___is_management_port_instance(((struct dm_data *)data)->dmmap_section))
				return 0;

			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "enabled", b ? "1" : "0");

			dmuci_get_value_by_section_string(bridge_args->config_section, "name", &device);
			if (DM_STRLEN(device) == 0)
				return 0;

			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "config", &config);
			if (DM_LSTRCMP(config, "wireless") == 0) {
				struct uci_section *wifi_iface_s = get_dup_section_in_config_opt("network", "interface", "device", device);
				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "network", b ? section_name(wifi_iface_s) : "");
			} else {
				char *port = NULL;

				dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "port", &port);
				if (DM_STRLEN(port) == 0)
					return 0;

				char *vlandevice = NULL;
				dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "8021q", &vlandevice);

				if (b) {
					add_port_to_bridge_sections(bridge_args->config_section, bridge_args->dmmap_section, port);
					if (!DM_STRCMP(vlandevice, "1")) {
						// last two args tell if the lower layer is another bridge port
						// in that case possibly 8021ad section has to be created and added
						// following also removes bridge-vlan section if present
						add_port_to_vlans((struct dm_data *)data, port, instance, 1, 1);
					} else {
						add_port_to_vlans((struct dm_data *)data, port, instance, 1, 0);
					}
				} else {
					remove_port_from_bridge_sections(bridge_args->config_section, bridge_args->dmmap_section, port);
					// last argument is flag to indicate that this is not a deletion
					remove_port_from_vlans((char *)(bridge_args->additional_data), bridge_args->config_section, port, 0);
				}
			}
			return 0;
	}
	return 0;
}

static int get_BridgingBridgePort_Status(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *port = NULL;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "port", &port);
	return get_net_device_status(port, value);
}

static int get_BridgingBridgePort_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return bbf_get_alias(ctx, ((struct dm_data *)data)->dmmap_section, "bridge_port_alias", instance, value);
}

static int set_BridgingBridgePort_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	return bbf_set_alias(ctx, ((struct dm_data *)data)->dmmap_section, "bridge_port_alias", instance, value);
}

static int get_BridgingBridgePort_Name(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "port", value);
	return 0;
}

static int get_BridgingBridgePort_LowerLayers(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	if (bridging___is_management_port_instance(((struct dm_data *)data)->dmmap_section)) {
		struct dm_data *bridge_args = ((struct dm_data *)data)->additional_data;
		struct uci_section *port_s = NULL;
		char buf[1024] = {0};
		char br_port_path[64] = {0};

		snprintf(br_port_path, sizeof(br_port_path), "Device.Bridging.Bridge.%s.Port.", (char *)bridge_args->additional_data);

		uci_path_foreach_option_eq(bbfdm, "dmmap_bridge_port", "bridge_port", "br_inst", (char *)bridge_args->additional_data, port_s) {
			char *mg_port = NULL;
			char *port = NULL;

			dmuci_get_value_by_section_string(port_s, "management", &mg_port);
			if (DM_LSTRCMP(mg_port, "1") == 0)
				continue;

			dmuci_get_value_by_section_string(port_s, "port", &port);

			bbfdm_get_references(ctx, MATCH_ALL, br_port_path, "Name", port, buf, sizeof(buf));
		}

		*value = dmstrdup(buf);
	} else {
		dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "LowerLayers", value);

		if ((*value)[0] == '\0') {
			char *type = NULL, *port = NULL, *config = NULL;

			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "type", &type);
			if (DM_STRCMP(type, "34984") == 0)
				return 0;

			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "port", &port);
			if (DM_STRLEN(port) == 0)
				return 0;

			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "config", &config);

			if (DM_LSTRCMP(config, "network") == 0) {
				_bbfdm_get_references(ctx, "Device.Ethernet.Interface.", "Name", port, value);
			} else {
				struct uci_section *iface_s = get_dup_section_in_config_opt("wireless", "wifi-iface", "ifname", port);
				if (iface_s != NULL) {
					char *ssid = NULL, *device = NULL, *name = NULL;

					dmuci_get_value_by_section_string(iface_s, "device", &device);
					dmuci_get_value_by_section_string(iface_s, "ssid", &ssid);

					struct uci_section *ssid_s = get_section_in_dmmap_with_options_eq("dmmap_wireless", "ssid", "device", device, "ssid", ssid);

					dmuci_get_value_by_section_string(ssid_s, "name", &name);

					_bbfdm_get_references(ctx, "Device.WiFi.SSID.", "Name", name, value);
				}
			}

			// Store LowerLayers value
			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "LowerLayers", *value);
		}
	}

	return 0;
}

static int set_BridgingBridgePort_LowerLayers(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct dm_data *bridge_args = ((struct dm_data *)data)->additional_data;
	char *allowed_objects[] = {
			"Device.Ethernet.Interface.",
			"Device.WiFi.SSID.",
			"Device.Bridging.Bridge.*.Port.",
			NULL
	};
	struct dm_reference reference = {0};
	char *enable = NULL, *port = NULL, *type = NULL;
	bool bridge_port = false;

	bbfdm_get_reference_linker(ctx, value, &reference);

	if (bridging___is_management_port_instance(((struct dm_data *)data)->dmmap_section))
		return 0;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string_list(ctx, reference.path, -1, -1, 1024, -1, -1, NULL, NULL))
				return FAULT_9007;

			if (dm_validate_allowed_objects(ctx, &reference, allowed_objects))
				return FAULT_9007;

			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "type", &type);
			if (DM_LSTRNCMP(value, "Device.Bridging.Bridge.", 23) == 0 && DM_STRCMP(type, "34984") != 0)
				return FAULT_9007;

			return 0;
		case VALUESET:
			if (DM_LSTRNCMP(value, "Device.Bridging.Bridge.", 23) == 0) {
				bridge_port = true;
			}

			// Store LowerLayers value under dmmap section
			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "LowerLayers", reference.path);

			// Update config section on dmmap_bridge_port if the linker is wirelss port or network port
			if (DM_LSTRNCMP(reference.path, "Device.WiFi.SSID.", 17) == 0) {
				dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "config", "wireless");
			} else {
				dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "config", "network");
			}

			// Get current port and enable options
			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "port", &port);
			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "enabled", &enable);

			if (DM_STRCMP(enable, "1") == 0) {
				// Remove port from port list interface
				remove_port_from_bridge_sections(bridge_args->config_section, bridge_args->dmmap_section, port);
				// the last flag indicates that this is a deletion (vs disabled)
				if (!bridge_port) {
					remove_port_from_vlans((char *)(bridge_args->additional_data), bridge_args->config_section, port, 0);
				}
			}

			// Update port option in dmmap
			char dest_8021q_port[32] = {0};

			if (bridge_port) {
				// the reason we dont pass the reference object directly
				// is that it is not available everywhere this function is called
				get_8021q_port(dest_8021q_port, sizeof(dest_8021q_port), reference.path, reference.value);
				if (DM_STRLEN(dest_8021q_port)) {
					dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "port", dest_8021q_port);
				}

				// set 8021q to let future use of this port be done accordingly
				dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "8021q", "1");
			} else {
				dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "8021q", "0");
				dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "port", reference.value);
			}

			bool port_enabled = 0;
			if (DM_STRCMP(enable, "1") == 0) {
				// needed for the function that updates vlans
				port_enabled = 1;
				// Add port to ports list
				if (bridge_port && DM_STRLEN(dest_8021q_port)) {
					add_port_to_bridge_sections(bridge_args->config_section, bridge_args->dmmap_section, dest_8021q_port);
				} else {
					add_port_to_bridge_sections(bridge_args->config_section, bridge_args->dmmap_section, reference.value);
				}
			}

			// if we have come so far and it is another bridge's port, then we need to
			// create a device section with type 8021ad for this port
			// this port should be eth0 for example
			// and add the created device's name to this bridge's ports list
			// assume bridge port was already added (example eth2)
			if (bridge_port && DM_STRLEN(dest_8021q_port)) {
				add_port_to_vlans((struct dm_data *)data, dest_8021q_port, instance, port_enabled, 1);
			} else {
				// update port in dmmap vlanport sections and network uci also if needed
				add_port_to_vlans((struct dm_data *)data, reference.value, instance, port_enabled, 0);
			}
			return 0;
		}
	return 0;
}

static int get_BridgingBridgePort_ManagementPort(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "management", value);
	return 0;
}

static int set_BridgingBridgePort_ManagementPort(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct dm_data *bridge_args = ((struct dm_data *)data)->additional_data;
	char *bridge_name = NULL;
	bool b;

	string_to_bool(value, &b);

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;

			if (b && get_section_in_dmmap_with_options_eq("dmmap_bridge_port", "bridge_port", "br_inst", (char *)bridge_args->additional_data, "management", "1"))
				return FAULT_9007;

			return 0;
		case VALUESET:
			dmuci_get_value_by_section_string(bridge_args->config_section, "name", &bridge_name);

			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "management", b ? "1" : "0");
			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "port", b ? bridge_name : "");

			// Update Ethernet Link instance if exists
			if (b) ethernet___Update_Link_Layer(refparam, bridge_name);
			return 0;
	}
	return 0;
}

static int get_BridgingBridgePort_PVID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->dmmap_section, "pvid", "1");
	return 0;
}

static int set_BridgingBridgePort_PVID(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct dm_data *bridge_args = ((struct dm_data *)data)->additional_data;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_int(ctx, value, RANGE_ARGS{{"1","4094"}}, 1))
				return FAULT_9007;

			if (bridging___is_management_port_instance(((struct dm_data *)data)->dmmap_section))
				return 0;

			return 0;
		case VALUESET:
			if (bridging___is_management_port_instance(((struct dm_data *)data)->dmmap_section))
				return 0;

			// set pvid in bridging.port dmmap section
			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "pvid", value);
			// find all the vlanport sections for this bridge.port
			// for all vlanport sections of this bridge
			struct uci_section *vlanport_dmmap = NULL;
			uci_path_foreach_option_eq(bbfdm, "dmmap_bridge_vlanport", "bridge_vlanport", "br_inst",
							(char *)(bridge_args->additional_data), vlanport_dmmap) {

				char *bridge_port_instance = NULL;
				dmuci_get_value_by_section_string(vlanport_dmmap, "bridge_port_instance", &bridge_port_instance);

				// if vlanport's 'bridge_port_instance matches this instance
				if (!DM_STRCMP(bridge_port_instance, instance)) {
					char *port_str = NULL;
					dmuci_get_value_by_section_string(vlanport_dmmap, "port_str", &port_str);

					// if enabled is not 0, meaning vlanport is enabled
					// we can just use port_str to check if both vlanport and bridge.port are enabled
					if (DM_STRLEN(port_str)) {
						// if existing pvid == vlanid and bridge_vlan_section is set
						// then we need to change in bridge_vlan section in network uci
						char *pvid = NULL;
						char *vlan = NULL;
						char *bridge_vlan_section = NULL;

						dmuci_get_value_by_section_string(vlanport_dmmap, "vlan", &vlan);
						dmuci_get_value_by_section_string(vlanport_dmmap, "pvid", &pvid);
						dmuci_get_value_by_section_string(vlanport_dmmap, "bridge_vlan_section", &bridge_vlan_section);
						// if bridge vlan section exists and
						// the new pvid matches this vlan, or previously pvid and vlanid were matching
						// then port_str needs updation
						// then we remove existing port_str from bridge-vlan in network uci
						// and add the new one in network uci and vlanport dmmap
						if (DM_STRLEN(bridge_vlan_section) && (!DM_STRCMP(value, vlan) || !DM_STRCMP(pvid, vlan))) {
							dmuci_del_list_value("network", bridge_vlan_section, "ports", port_str);
							// set the new pvid in vlanport dmmap
							dmuci_set_value_by_section(vlanport_dmmap, "pvid", value);
							char new_port_str[32] = {0};
							// generate new port str, set it in bridge_vlan_section
							get_vlan_port_str(vlanport_dmmap, NULL, new_port_str, sizeof(new_port_str));
							// set it in network uci
							dmuci_add_list_value("network", bridge_vlan_section, "ports", new_port_str);

							// set it in dmmap
							dmuci_set_value_by_section(vlanport_dmmap, "port_str", new_port_str);
							// nothing more to do
							continue;
						}
					}

					// set the new pvid in vlanport dmmap
					dmuci_set_value_by_section(vlanport_dmmap, "pvid", value);
				}
			}
	}
	return 0;
}

static int get_BridgingBridgePort_TPID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *type = NULL;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "type", &type);
	if (DM_LSTRCMP(type, "8021q") == 0)
		*value = dmstrdup("33024");
	else if (DM_LSTRCMP(type, "8021ad") == 0)
		*value = dmstrdup("34984");
	else
		*value = dmstrdup("37120");
	return 0;
}

static int set_BridgingBridgePort_TPID(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,NULL}}, 1))
				return FAULT_9007;

			return 0;
		case VALUESET:
			if (bridging___is_management_port_instance(((struct dm_data *)data)->dmmap_section))
				return 0;

			if (DM_LSTRCMP(value, "33024") == 0)
				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "type", "8021q");
			else if (DM_LSTRCMP(value, "34984") == 0)
				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "type", "8021ad");

			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "type", value);
			return 0;
	}
	return 0;
}

static int br_get_sysfs(const struct dm_data *bridge_sec, const char *name, char **value)
{
	char *device = NULL;

	dmuci_get_value_by_section_string(bridge_sec->config_section, "ifname", &device);
	return get_net_device_sysfs(device, name, value);
}

static int br_get_ubus_eth(const struct dm_data *bridge_sec, const char *name, char **value)
{
	json_object *res = NULL;
	char *device = NULL, *config = NULL;

	DM_ASSERT(bridge_sec, *value = dmstrdup("0"));
	dmuci_get_value_by_section_string(bridge_sec->config_section, "ifname", &device);
	dmuci_get_value_by_section_string(bridge_sec->dmmap_section, "config", &config);

	if (DM_LSTRCMP(config, "network") == 0) {
		*value = dmstrdup("0");
	} else {
		char object[32];

		snprintf(object, sizeof(object), "wifi.ap.%s", device);
		dmubus_call(object, "stats", UBUS_ARGS{0}, 0, &res);
	}

	DM_ASSERT(res, *value = dmstrdup("0"));
	*value = dmjson_get_value(res, 1, name);
	return 0;
}

/*#Device.Bridging.Bridge.{i}.Port.{i}.Stats.BytesSent!SYSFS:/sys/class/net/@Name/statistics/tx_bytes*/
static int get_BridgingBridgePortStats_BytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return br_get_sysfs(data, "statistics/tx_bytes", value);
}

/*#Device.Bridging.Bridge.{i}.Port.{i}.Stats.BytesSent!SYSFS:/sys/class/net/@Name/statistics/rx_bytes*/
static int get_BridgingBridgePortStats_BytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return br_get_sysfs(data, "statistics/rx_bytes", value);
}

/*#Device.Bridging.Bridge.{i}.Port.{i}.Stats.PacketsSent!SYSFS:/sys/class/net/@Name/statistics/tx_packets*/
static int get_BridgingBridgePortStats_PacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return br_get_sysfs(data, "statistics/tx_packets", value);
}

/*#Device.Bridging.Bridge.{i}.Port.{i}.Stats.PacketsReceived!SYSFS:/sys/class/net/@Name/statistics/rx_packets*/
static int get_BridgingBridgePortStats_PacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return br_get_sysfs(data, "statistics/rx_packets", value);
}

/*#Device.Bridging.Bridge.{i}.Port.{i}.Stats.ErrorsSent!SYSFS:/sys/class/net/@Name/statistics/tx_errors*/
static int get_BridgingBridgePortStats_ErrorsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return br_get_sysfs(data, "statistics/tx_errors", value);
}

/*#Device.Bridging.Bridge.{i}.Port.{i}.Stats.ErrorsReceived!SYSFS:/sys/class/net/@Name/statistics/rx_errors*/
static int get_BridgingBridgePortStats_ErrorsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return br_get_sysfs(data, "statistics/rx_errors", value);
}

static int get_BridgingBridgePortStats_UnicastPacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return br_get_ubus_eth(data, "tx_unicast_packets", value);
}

static int get_BridgingBridgePortStats_UnicastPacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return br_get_ubus_eth(data, "rx_unicast_packets", value);
}

/*#Device.Bridging.Bridge.{i}.Port.{i}.Stats.DiscardPacketsSent!SYSFS:/sys/class/net/@Name/statistics/tx_dropped*/
static int get_BridgingBridgePortStats_DiscardPacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return br_get_sysfs(data, "statistics/tx_dropped", value);
}

/*#Device.Bridging.Bridge.{i}.Port.{i}.Stats.DiscardPacketsReceived!SYSFS:/sys/class/net/@Name/statistics/rx_dropped*/
static int get_BridgingBridgePortStats_DiscardPacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return br_get_sysfs(data, "statistics/rx_dropped", value);
}

static int get_BridgingBridgePortStats_MulticastPacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return br_get_ubus_eth(data, "tx_multicast_packets", value);
}

/*#Device.Bridging.Bridge.{i}.Port.{i}.Stats.MulticastPacketsReceived!SYSFS:/sys/class/net/@Name/statistics/multicast*/
static int get_BridgingBridgePortStats_MulticastPacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return br_get_sysfs(data, "statistics/multicast", value);
}

static int get_BridgingBridgePortStats_BroadcastPacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return br_get_ubus_eth(data, "tx_broadcast_packets", value);
}

static int get_BridgingBridgePortStats_BroadcastPacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return br_get_ubus_eth(data, "rx_broadcast_packets", value);
}

static int get_BridgingBridgePortStats_UnknownProtoPacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return br_get_ubus_eth(data, "rx_unknown_packets", value);
}

static int get_BridgingBridgeVLAN_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "enabled", "1");
	return 0;
}

static int set_BridgingBridgeVLAN_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "enabled",  b ? "1" : "0");
			return 0;
	}
	return 0;
}

static int get_BridgingBridgeVLAN_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return bbf_get_alias(ctx, ((struct dm_data *)data)->dmmap_section, "bridge_vlan_alias", instance, value);
}

static int set_BridgingBridgeVLAN_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	return bbf_set_alias(ctx, ((struct dm_data *)data)->dmmap_section, "bridge_vlan_alias", instance, value);
}

static int get_BridgingBridgeVLAN_Name(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "section_name", value);
	return 0;
}

static int set_BridgingBridgeVLAN_Name(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 64, NULL, NULL))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "section_name", value);
			return 0;
	}
	return 0;
}

static int get_BridgingBridgeVLAN_VLANID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "vlan", value);
	return 0;
}

static int set_BridgingBridgeVLAN_VLANID(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_int(ctx, value, RANGE_ARGS{{"1","4094"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
		{
			struct uci_section *vlanport_section = NULL;
			struct uci_section *bridge_vlan_section = ((struct dm_data *)data)->config_section;
			struct dm_data *bridge_args = ((struct dm_data *)data)->additional_data;

			// find all vlanport sections with this bridge_vlan_section
			uci_path_foreach_option_eq(bbfdm, "dmmap_bridge_vlanport", "bridge_vlanport", "bridge_vlan_instance", instance,
							vlanport_section) {
				// if port is set and vlanport is enabled
				// if pvid matches with new value
				// then get port str based on old value
				// delete it from this bridge_vlan_section
				// get port_str based on new value
				// set it in this bridge_vlan_section
				dmuci_set_value_by_section(vlanport_section, "vlan", value);

				char *port = NULL;
				dmuci_get_value_by_section_string(vlanport_section, "port", &port);

				char *enabled = NULL;
				dmuci_get_value_by_section_string(vlanport_section, "enabled", &enabled);

				char *br_inst = (char *)(char *)(bridge_args->additional_data);

				char *vlanport_br_inst = NULL;
				dmuci_get_value_by_section_string(vlanport_section, "br_inst", &vlanport_br_inst);

				// if port is set and enabled is not 0 vlanport_br_inst and br_inst match
				if (DM_STRLEN(port) && DM_STRCMP(enabled, "0") && !DM_STRCMP(vlanport_br_inst, br_inst)) {
					char *vlandevice = NULL;
					dmuci_get_value_by_section_string(vlanport_section, "8021q", &vlandevice);

					if (DM_STRCMP(vlandevice, "1") == 0) {
						// this is a port with vlan device section
						// if port_str is set then delete it from bridge ports list
						// create new port_str based on port and value set
						// set it in bridge ports list
						// set it in vlanport
						// delete bridge_vlan_section if present
						struct uci_section *bridge_section = bridge_args->config_section;
						char *port_str = NULL;

						dmuci_get_value_by_section_string(vlanport_section, "port_str", &port_str);
						if (DM_STRLEN(port_str)) {
							dmuci_del_list_value_by_section(bridge_section, "ports", port_str);
						}

						char *vlanport = NULL;
						dmuci_get_value_by_section_string(vlanport_section, "port", &vlanport);
						if (DM_STRLEN(vlanport)) {
							char new_port[32] = {0};

							snprintf(new_port, sizeof(new_port), "%s.%s", vlanport, value);

							// set it in bridge uci and vlanport dmmap section
							dmuci_add_list_value_by_section(bridge_section, "ports", new_port);

							dmuci_set_value_by_section(vlanport_section, "port_str", new_port);
						}

						char *bridge_vlan_section_name = NULL;
						dmuci_get_value_by_section_string(vlanport_section, "bridge_vlan_section",
											&bridge_vlan_section_name);

						if (DM_STRLEN(bridge_vlan_section_name)) {
							struct uci_section *bridge_vlan_sec = get_origin_section_from_config("network",
													"bridge-vlan",
													bridge_vlan_section_name);
							if (bridge_vlan_sec) {
								dmuci_delete_by_section(bridge_vlan_sec, NULL, NULL);
							}
						}
					} else {
						char *pvid = NULL;
						dmuci_get_value_by_section_string(vlanport_section, "pvid", &pvid);

						// get port str and delete it from bridge_vlan_section
						char port_str[32] = {0};
						get_vlan_port_str(vlanport_section, port, port_str, sizeof(port_str));

						dmuci_del_list_value_by_section(bridge_vlan_section, "ports", port_str);
						// set the vlan here too so that new port str can be calculated
						dmuci_set_value_by_section(vlanport_section, "vlan", value);

						get_vlan_port_str(vlanport_section, port, port_str, sizeof(port_str));

						dmuci_add_list_value_by_section(bridge_vlan_section, "ports", port_str);
						// update port str in dmmap too
						dmuci_set_value_by_section(vlanport_section, "port_str", port_str);
					}
				}
			}
			// update in bridge-vlan section
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "vlan", value);
			// needed to map VLANID to dmmap section
			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "vlan", value);
			return 0;
		}
	}
	return 0;
}

static int get_BridgingBridgeVLANPort_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->dmmap_section, "enabled", "1");
	return 0;
}

static int set_BridgingBridgeVLANPort_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct dm_data *bridge_args = ((struct dm_data *)data)->additional_data;
	char *br_inst = (char *)(bridge_args->additional_data);

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
		{
			// get current value
			bool new_enabled, current_enabled;
			struct uci_section *vlanport_section = ((struct dm_data *)data)->dmmap_section;
			char *current_enabled_str = dmuci_get_value_by_section_fallback_def(vlanport_section, "enabled", "1");

			// get in bool new enabled value
			string_to_bool(value, &new_enabled);
			// get in bool current enabled value
			string_to_bool(current_enabled_str, &current_enabled);

			// if current and new are different
			if (current_enabled != new_enabled) {
				char *vlandevice = NULL;
				dmuci_get_value_by_section_string(vlanport_section, "8021q", &vlandevice);

				if (!DM_STRCMP(vlandevice, "1")) {
					struct uci_section *bridge_section = bridge_args->config_section;
					struct uci_section *bridge_dmmap_section = bridge_args->dmmap_section;
					// vlanport is being enabled
					if (new_enabled) {
						char *vlan = NULL;
						dmuci_get_value_by_section_string(vlanport_section, "vlan", &vlan);
						char *port = NULL;
						dmuci_get_value_by_section_string(vlanport_section, "port", &port);

						if (DM_STRLEN(vlan) && DM_STRLEN(port)) {
							// port here is something like eth0.100
							// remove it from bridge ports list
							dmuci_del_list_value_by_section(bridge_section, "ports", port);

							char port_str[32] = {0};
							create_8021ad_device_section(port_str, sizeof(port_str), port, vlan, br_inst,
									instance, vlanport_section);

							// port_str is something like eth0.100.200
							// set it in vlanport dmmap
							// set it in bridge's ports list
							dmuci_set_value_by_section(vlanport_section, "port_str", port_str);
							dmuci_add_list_value_by_section(bridge_section, "ports", port_str);
							dmuci_add_list_value_by_section(bridge_dmmap_section, "ports", port_str);
						}
					} else {
						char *port_str = NULL;
						dmuci_get_value_by_section_string(vlanport_section, "port_str", &port_str);

						if (DM_STRLEN(port_str)) {
							// remove from vlanport, bridge config, bridge dmmap
							dmuci_set_value_by_section(vlanport_section, "port_str", "");
							dmuci_del_list_value_by_section(bridge_section, "ports", port_str);
							dmuci_del_list_value_by_section(bridge_dmmap_section, "ports", port_str);
						}
					}
				} else {
					// if bridge_vlan_section is set
					char *bridge_vlan_section = NULL;
					dmuci_get_value_by_section_string(vlanport_section, "bridge_vlan_section", &bridge_vlan_section);
					if (DM_STRLEN(bridge_vlan_section)) {
						// get the port
						char *port = NULL;
						// the presence of port indicates that Bridging.Bridge.Port is enabled
						dmuci_get_value_by_section_string(vlanport_section, "port", &port);
						// add it to bridge_vlan_section
						if (DM_STRLEN(port)) {
							// if new is to enable the port
							char port_str[32] = {0};
							get_vlan_port_str(vlanport_section, port, port_str, sizeof(port_str));

							if (new_enabled) {
								dmuci_add_list_value("network", bridge_vlan_section, "ports", port_str);
							} else {
								dmuci_del_list_value("network", bridge_vlan_section, "ports", port_str);
							}
						}
					}
				}

				// in any case we have to set it in dmmap
				dmuci_set_value_by_section(vlanport_section, "enabled", value);
			}
			break;
		}
	}
	return 0;
}

static int get_BridgingBridgeVLANPort_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return bbf_get_alias(ctx, ((struct dm_data *)data)->dmmap_section, "bridge_vlanport_alias", instance, value);
}

static int set_BridgingBridgeVLANPort_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	return bbf_set_alias(ctx, ((struct dm_data *)data)->dmmap_section, "bridge_vlanport_alias", instance, value);
}

static int get_BridgingBridgeVLANPort_VLAN(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char br_vlan_path[64] = {0};
	char *vlan = NULL;
	struct dm_data *bridge_args = ((struct dm_data *)data)->additional_data;
        char *br_inst = (char *)(bridge_args->additional_data);
	struct uci_section *vlanport_dmmap_sec = ((struct dm_data *)data)->dmmap_section;
	dmuci_get_value_by_section_string(vlanport_dmmap_sec, "vlan", &vlan);

	if (!DM_STRLEN(br_inst)) {
		*value = dmstrdup("");
		return 0;
	}

	// we can extract vlan instance from vlanport dmmap section but we do not do that
	// because we have to use the reference mechanism here
	snprintf(br_vlan_path, sizeof(br_vlan_path), "Device.Bridging.Bridge.%s.VLAN.", br_inst);

	_bbfdm_get_references(ctx, br_vlan_path, "VLANID", vlan, value);

	return 0;
}

static int set_BridgingBridgeVLANPort_VLAN(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char lower_layer_path[256] = {0};
	char *allowed_objects[] = {
			lower_layer_path,
			NULL
	};
	struct dm_reference reference = {0};

	bbfdm_get_reference_linker(ctx, value, &reference);

	struct dm_data *bridge_args = ((struct dm_data *)data)->additional_data;
	char *br_inst = (char *)(bridge_args->additional_data);

	snprintf(lower_layer_path, sizeof(lower_layer_path), "Device.Bridging.Bridge.%s.VLAN.", br_inst);

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, reference.path, -1, 256, NULL, NULL)) {
				return FAULT_9007;
			}

			if (dm_validate_allowed_objects(ctx, &reference, allowed_objects)) {
				return FAULT_9007;
			}

			return 0;
		case VALUESET:
		{
			// if this is a 8021q type vlanport section
			// then update vlan_inst
			// delete the bridge_vlan section from network uci
			// if new vlan is set
			// then update it in vlanport dmmap
			// and get the port from vlanport dmmap
			// delete it from network uci section of bridge
			// generate the new port name
			// set it in bridge's port list
			// if this is not a 8021q type vlanport section
			// if port is set
			// get the current VLAN section
			// delete the port from there
			// find the new VLAN section
			// add the port there
			// set it in dmmap
			char *vlanport = NULL;
			char *new_bridge_vlan = NULL;
			struct uci_section *vlanport_dmmap_section = ((struct dm_data *)data)->dmmap_section;
			struct uci_section *new_bridge_vlan_dmmap = NULL;

			char *vlandevice = NULL;
			dmuci_get_value_by_section_string(vlanport_dmmap_section, "8021q", &vlandevice);

			// get pointer to dmmap section for new vlan object
			// we could have got the network uci section directly with the bridge name and vlan
			// but we need the dmmap section of bridge-vlan because section might be created first
			// and vid might be added later
			char *bridge_vlan_instance = NULL;

			// get the instance in the path that was passed
			bridge_vlan_instance = DM_STRRCHR(reference.path, '.');
			if (!bridge_vlan_instance) {
				return FAULT_9002;
			}

			bridge_vlan_instance =  bridge_vlan_instance + 1;

			new_bridge_vlan_dmmap = get_section_in_dmmap_with_options_eq("dmmap_bridge_vlan", "bridge-vlan",
										"br_inst", br_inst,
										"bridge_vlan_instance", bridge_vlan_instance);
			// if no dmmap section then return
			if (!new_bridge_vlan_dmmap) {
				return FAULT_9002;
			}

			// get network uci section from new_bridge_vlan_dmmap
			dmuci_get_value_by_section_string(new_bridge_vlan_dmmap, "section_name", &new_bridge_vlan);

			// if no name for new vlan section then return
			if (!new_bridge_vlan) {
				return FAULT_9002;
			}

			if (!DM_STRCMP(vlandevice, "1")) {
				// delete this bridge vlan section in network uci
				char *bridge_vlan_section_name = NULL;
				struct uci_section *bridge_vlan_config_s = NULL;

				dmuci_get_value_by_section_string(new_bridge_vlan_dmmap, "section_name", &bridge_vlan_section_name);
				if (DM_STRLEN(bridge_vlan_section_name)) {
					bridge_vlan_config_s = get_origin_section_from_config("network", "bridge-vlan", bridge_vlan_section_name);
					if (bridge_vlan_config_s) {
						dmuci_delete_by_section(bridge_vlan_config_s, NULL, NULL);
					}
				} else {
					return FAULT_9002;
				}

				// check if new vlan is set
				if (DM_STRLEN(reference.value)) {
					// then get the port from dmmap vlanport
					dmuci_get_value_by_section_string(vlanport_dmmap_section, "port", &vlanport);

					char *enabled = NULL;
					dmuci_get_value_by_section_string(vlanport_dmmap_section, "enabled", &enabled);

					if (DM_STRLEN(vlanport) && DM_STRCMP(enabled, "0")) {
						// delete it from bridge's ports list
						struct uci_section *bridge_section = bridge_args->config_section;
						struct uci_section *bridge_dmmap_section = bridge_args->dmmap_section;
						// get ports list
						struct uci_list *ports_list = NULL;
						dmuci_get_value_by_section_list(bridge_section, "ports", &ports_list);

						if (ports_list) {
							// loop over bridge's ports list
							struct uci_element *e = NULL;
							uci_foreach_element(ports_list, e) {
								if (DM_STRSTR(e->name, vlanport)) {
									dmuci_del_list_value_by_section(bridge_section, "ports", e->name);
									dmuci_del_list_value_by_section(bridge_dmmap_section, "ports", e->name);
									break;
								}
							}
						}

						// and then set the new port
						char new_port[32] = {0};

						// update it in 8021ad section
						char *qinq_section_name = NULL;
						dmuci_get_value_by_section_string(vlanport_dmmap_section, "qinq_section_name", &qinq_section_name);
						if (DM_STRLEN(qinq_section_name)) {
							// qinq section already exists, update it
							snprintf(new_port, sizeof(new_port), "%s.%s", vlanport, reference.value);

							struct uci_section *qinq_section = NULL;
							qinq_section = get_origin_section_from_config("network", "device", qinq_section_name);

							if (qinq_section) {
								dmuci_set_value_by_section(qinq_section, "name", new_port);
								dmuci_set_value_by_section(qinq_section, "vid", reference.value);
							}
						} else {
							// port here is something like eth0.100
							// remove it from bridge ports list
							dmuci_del_list_value_by_section(bridge_section, "ports", vlanport);

							// qinq section does not exist, create it
							create_8021ad_device_section(new_port, sizeof(new_port), vlanport, reference.value, br_inst, instance, vlanport_dmmap_section);
						}

						// set it in bridge section
						dmuci_add_list_value_by_section(bridge_section, "ports", new_port);
						dmuci_add_list_value_by_section(bridge_dmmap_section, "ports", new_port);
						// set it in dmmap vlanport
						dmuci_set_value_by_section(vlanport_dmmap_section, "port_str", new_port);
					}
				}
			} else {
				// get the ifname for current VLANPort object
				dmuci_get_value_by_section_string(vlanport_dmmap_section, "port", &vlanport);

				char *enabled = NULL;
				dmuci_get_value_by_section_string(vlanport_dmmap_section, "enabled", &enabled);

				// if VLANPort found and enabled is not 0
				if (DM_STRLEN(vlanport) && DM_STRCMP(enabled, "0")) {
					char *current_bridge_vlan = NULL;
					char port_str[32] = {0};

					get_vlan_port_str(vlanport_dmmap_section, vlanport, port_str, sizeof(port_str));
					// get current vlan section
					dmuci_get_value_by_section_string(vlanport_dmmap_section, "bridge_vlan_section", &current_bridge_vlan);

					// if current_bridge_vlan was set, then delete the port from there
					if (DM_STRLEN(current_bridge_vlan)) {
						dmuci_del_list_value("network", current_bridge_vlan, "ports", port_str);
					}

					// if new_bridge_vlan was found, set the port there
					if (DM_STRLEN(new_bridge_vlan)) {
						dmuci_add_list_value("network", new_bridge_vlan, "ports", port_str);
					}
				}
			}

			// Store VLAN value under dmmap section
			dmuci_set_value_by_section(vlanport_dmmap_section, "bridge_vlan_section", new_bridge_vlan);
			dmuci_set_value_by_section(vlanport_dmmap_section, "vlan", reference.value);
			dmuci_set_value_by_section(vlanport_dmmap_section, "bridge_vlan_instance", bridge_vlan_instance);
		}
	}
	return 0;
}

static int get_BridgingBridgeVLANPort_Port(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *vlanport_dmmap_section = ((struct dm_data *)data)->dmmap_section;

	struct dm_data *bridge_args = ((struct dm_data *)data)->additional_data;
        char *br_inst = (char *)(bridge_args->additional_data);

	char br_port_path[128] = {0};
	char *port = NULL;

	snprintf(br_port_path, sizeof(br_port_path), "Device.Bridging.Bridge.%s.Port.", br_inst);

	// get value of port which will be of the form 'eth4:t'
	dmuci_get_value_by_section_string(vlanport_dmmap_section, "port", &port);
	if (!DM_STRLEN(port))
		return 0;

	//remove the colon
	char *colon = NULL;
	// find the colon
	colon = DM_STRSTR(port, ":");
	// replace it with null
	if (colon)
		*colon = '\0';

	_bbfdm_get_references(ctx, br_port_path, "Name", port, value);

	return 0;
}

static void delete_port_from_bridge(struct uci_section *bridge_section, char *port)
{
	if (!bridge_section) {
		return;
	}

	if (!DM_STRLEN(port)) {
		return;
	}

	struct uci_list *ports_list = NULL;
	struct uci_element *e = NULL;

	dmuci_get_value_by_section_list(bridge_section, "ports", &ports_list);
	uci_foreach_element(ports_list, e) {
		if (DM_STRSTR(e->name, port)) {
			dmuci_del_list_value_by_section(bridge_section, "ports", e->name);
			break;
		}
	}
}

static int set_BridgingBridgeVLANPort_Port(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct uci_section *vlanport_dmmap_section = ((struct dm_data *)data)->dmmap_section;
	struct dm_data *bridge_args = ((struct dm_data *)data)->additional_data;

	char *br_inst = (char *)(bridge_args->additional_data);

	char lower_layer_path[64] = {0};
	char *allowed_objects[] = {
		lower_layer_path,
		NULL
	};
	struct dm_reference reference = {0};

	bbfdm_get_reference_linker(ctx, value, &reference);

	snprintf(lower_layer_path, sizeof(lower_layer_path), "Device.Bridging.Bridge.%s.Port.", br_inst);

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, reference.path, -1, 256, NULL, NULL))
				return FAULT_9007;

			if (dm_validate_allowed_objects(ctx, &reference, allowed_objects))
				return FAULT_9007;

			return 0;
		case VALUESET:
		{
			// find the bridge vlan section for this vlanport
			// if bridge vlan section exists
			// get the ifname corresponding to the current port
			// delete it from bridge-vlan section
			// get the ifname for the new port
			// create the port string (append :t)
			// if the new port is enabled and this vlanport is enabled
			// add it to the bridge_vlan section
			// if bridge vlan section does not exist or port is not enabled
			// just save the port string to dmmap
			// and bridge port path too
			char *current_bridge_vlan = NULL;
			// get the current bridge vlan section
			dmuci_get_value_by_section_string(vlanport_dmmap_section, "bridge_vlan_section", &current_bridge_vlan);

			char *bridge_port_instance = NULL;
			// get the instance number for the port passed
			bridge_port_instance = DM_STRRCHR(reference.path, '.');
			if (!bridge_port_instance) {
				return FAULT_9002;
			}

			// skip the . to get only the numerical value
			bridge_port_instance = bridge_port_instance + 1;

			struct uci_section *dmmap_port_section = NULL;
			// get the Bridge.Port section of this bridge which has this bridge_port_instance
			dmmap_port_section = get_section_in_dmmap_with_options_eq("dmmap_bridge_port", "bridge_port",
					"br_inst", br_inst,
					"bridge_port_instance", bridge_port_instance);
			// section not found then return
			if (!dmmap_port_section) {
				return FAULT_9002;
			}

			char *pvid = NULL;
			dmuci_get_value_by_section_string(dmmap_port_section, "pvid", &pvid);

			if (DM_STRLEN(pvid)) {
				dmuci_set_value_by_section(vlanport_dmmap_section, "pvid", pvid);
			}

			char *port_enabled = NULL;
			// see if this Bridge.Port is enabled
			port_enabled = dmuci_get_value_by_section_fallback_def(dmmap_port_section, "enabled", "1");
			char *vlandevice = NULL;
			dmuci_get_value_by_section_string(dmmap_port_section, "8021q", &vlandevice);

			char *vlanport_enabled = NULL;
			// see if this Bridge.VLANPort is enabled
			vlanport_enabled = dmuci_get_value_by_section_fallback_def(vlanport_dmmap_section, "enabled", "1");

			// if Bridge.Port is enabled, then set it in dmmap to indicate that it is enabled
			if (!DM_STRCMP(port_enabled, "1")) {
				dmuci_set_value_by_section(vlanport_dmmap_section, "8021q", "1");
				dmuci_set_value_by_section(vlanport_dmmap_section, "port", reference.value);
				// if port is enabled
				// then first get the current port
				// delete it from bridge's ports list
				// set port instance in vlanport dmmap anyway
				if (!DM_STRCMP(vlanport_enabled, "1")) {
					// if underlying port is vlandevice
					// then set vlandevice in vlanport too
					// otherwise unset it
					// if vlan is set
					// then create 8021ad section using the supplied port
					// and vlan
					// and set it in vlanport
					// and set it in bridge ports list
					// also delete existing bridge-vlan section
					if (!DM_STRCMP(vlandevice, "1")) {
						struct uci_section *bridge_section = bridge_args->config_section;
						struct uci_section *bridge_dmmap_section = bridge_args->dmmap_section;

						delete_port_from_bridge(bridge_section, reference.value);
						delete_port_from_bridge(bridge_dmmap_section, reference.value);

						char *vlan = NULL;
						dmuci_get_value_by_section_string(vlanport_dmmap_section, "vlan", &vlan);
						if (DM_STRLEN(vlan)) {
							char port_str[32] = {0};
							create_8021ad_device_section(port_str, sizeof(port_str), reference.value, vlan, br_inst, instance, vlanport_dmmap_section);
							// set it in vlanport dmmap
							// set it in bridge's ports list
							dmuci_set_value_by_section(vlanport_dmmap_section, "port_str", port_str);
							dmuci_add_list_value_by_section(bridge_section, "ports", port_str);
							dmuci_add_list_value_by_section(bridge_dmmap_section, "ports", port_str);

							char *bridge_vlan_section_name = NULL;
							dmuci_get_value_by_section_string(vlanport_dmmap_section, "bridge_vlan_section",
												&bridge_vlan_section_name);

							if (DM_STRLEN(bridge_vlan_section_name)) {
								struct uci_section *bridge_vlan_section = get_origin_section_from_config(
														"network",
														"bridge-vlan",
														bridge_vlan_section_name);
								if (bridge_vlan_section) {
									dmuci_delete_by_section(bridge_vlan_section, NULL, NULL);
								}

								dmuci_set_value_by_section(vlanport_dmmap_section, "bridge_vlan_section", "");
							}
						}
					} else {
						dmuci_set_value_by_section(vlanport_dmmap_section, "8021q", "0");
						char port_str[32] = {0};

						if (DM_STRLEN(current_bridge_vlan)) {
							// if port was set then delete it from network uci
							char *current_port = NULL;
							// get the present value for port
							dmuci_get_value_by_section_string(vlanport_dmmap_section, "port", &current_port);

							if (DM_STRLEN(current_port)) {
								get_vlan_port_str(vlanport_dmmap_section, current_port, port_str, sizeof(port_str));

								dmuci_del_list_value("network", current_bridge_vlan, "ports", port_str);
							}
						}

						get_vlan_port_str(vlanport_dmmap_section, reference.value, port_str, sizeof(port_str));

						dmuci_set_value_by_section(vlanport_dmmap_section, "port_str", port_str);

						// if bridge vlan was set then set it in network uci too
						if (DM_STRLEN(current_bridge_vlan)) {
							dmuci_add_list_value("network", current_bridge_vlan, "ports", port_str);
						}
					}
				}
			}

			// Store bridge port instance under dmmap section in any case
			dmuci_set_value_by_section(vlanport_dmmap_section, "bridge_port_instance", bridge_port_instance);
			return 0;
		}
	}
	return 0;
}

static int get_BridgingBridgeVLANPort_Untagged(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->dmmap_section, "untagged", "1");
	return 0;
}

static int set_BridgingBridgeVLANPort_Untagged(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
		{
			struct uci_section *vlanport_dmmap = ((struct dm_data *)data)->dmmap_section;
			char *current_untagged = dmuci_get_value_by_section_fallback_def(vlanport_dmmap, "untagged", "1");

			// if current value and new value are different
			if (DM_STRCMP(current_untagged, value)) {
				// set new untagged
				dmuci_set_value_by_section(vlanport_dmmap, "untagged", value);

				// if bridge vlan is set for this vlanport
				char *bridge_vlan_section = NULL;
				dmuci_get_value_by_section_string(vlanport_dmmap, "bridge_vlan_section", &bridge_vlan_section);
				if (DM_STRLEN(bridge_vlan_section)) {
					// get the port
					char *port = NULL;
					dmuci_get_value_by_section_string(vlanport_dmmap, "port", &port);

					// get enabled
					char *enabled = NULL;
					dmuci_get_value_by_section_string(vlanport_dmmap, "enabled", &enabled);

					// if enabled not 0, remove old value and set new value
					if (DM_STRCMP(enabled, "0")) {
						struct uci_list *ports = NULL;
						dmuci_get_option_value_list("network", bridge_vlan_section, "ports", &ports);

						// loop over all ports
						struct uci_element *e = NULL;
						uci_foreach_element(ports, e) {
							// if this element has our port, delete it
							if (DM_STRSTR(e->name, port)) {
								dmuci_del_list_value("network", bridge_vlan_section, "ports", e->name);
							}
						}
						char port_str[32] = {0};
						get_vlan_port_str(vlanport_dmmap, port, port_str, sizeof(port_str));

						dmuci_add_list_value("network", bridge_vlan_section, "ports", port_str);
						dmuci_set_value_by_section(vlanport_dmmap, "port_str", port_str);
					}
				}
			}
		}
	}
	return 0;
}

static int get_BridgingBridgeProviderBridge_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "enable", "0");
	return 0;
}

static int set_BridgingBridgeProviderBridge_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action)	{
	case VALUECHECK:
		if (bbfdm_validate_boolean(ctx, value))
			return FAULT_9007;
		break;
	case VALUESET:
		string_to_bool(value, &b);
		dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "enable", b ? "1" : "0");
		break;
	}
	return 0;
}

static int get_BridgingBridgeProviderBridge_Status(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *name = NULL;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "name", &name);
	get_net_device_status(name, value);
	if (DM_STRCMP(*value, "Up") == 0) {
		*value = dmstrdup("Enabled");
	} else {
		*value = dmstrdup("Disabled");
	}
	return 0;
}

static int get_BridgingBridgeProviderBridge_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return bbf_get_alias(ctx, ((struct dm_data *)data)->config_section, "provider_bridge_alias", instance, value);
}

static int set_BridgingBridgeProviderBridge_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
	case VALUECHECK:
		if (bbfdm_validate_string(ctx, value, -1, 64, NULL, NULL))
			return FAULT_9007;
		break;
	case VALUESET:
		dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "provider_bridge_alias", value);
		break;
	}
	return 0;
}

static int get_BridgingBridgeProviderBridge_SVLANcomponent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *br_inst = NULL;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "svlan_br_inst", &br_inst);
	if (DM_STRLEN(br_inst))
		dmasprintf(value, "Device.Bridging.Bridge.%s", br_inst);
	return 0;
}

static int set_BridgingBridgeProviderBridge_SVLANcomponent(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *allowed_objects[] = {"Device.Bridging.Bridge.", NULL};
	struct dm_reference reference = {0};

	bbfdm_get_reference_linker(ctx, value, &reference);

	switch (action)	{
	case VALUECHECK:
		if (bbfdm_validate_string(ctx, reference.path, -1, 256, NULL, NULL))
			return FAULT_9007;

		if (dm_validate_allowed_objects(ctx, &reference, allowed_objects))
			return FAULT_9007;

		break;
	case VALUESET:
		set_Provider_bridge_component(refparam, ctx, data, instance, reference.value, "SVLAN");
		break;
	}
	return 0;
}

static int get_BridgingBridgeProviderBridge_CVLANcomponents(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_list *cvlan_list = NULL;
	struct uci_element *e = NULL;
	char cvlan_buf[2048] = {0};
	unsigned pos = 0;

	dmuci_get_value_by_section_list(((struct dm_data *)data)->config_section, "cvlan_br_inst", &cvlan_list);
	if (cvlan_list == NULL)
		return 0;

	cvlan_buf[0] = 0;
	/* Traverse each list value and create comma separated bridge path */
	uci_foreach_element(cvlan_list, e) {
		pos += snprintf(&cvlan_buf[pos], sizeof(cvlan_buf) - pos, "Device.Bridging.Bridge.%s,", e->name);
	}

	if (pos)
		cvlan_buf[pos - 1] = 0;

	*value = dmstrdup(cvlan_buf);
	return 0;
}

static int set_BridgingBridgeProviderBridge_CVLANcomponents(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *allowed_objects[] = {"Device.Bridging.Bridge.", NULL};
	struct dm_reference reference = {0};
	char *pch = NULL, *pchr = NULL;
	char buf[512] = {0};

	DM_STRNCPY(buf, value, sizeof(buf));

	switch (action)	{
		case VALUECHECK:
			// Validate each item in list and Check if bridge is present
			for (pch = strtok_r(buf, ",", &pchr); pch != NULL; pch = strtok_r(NULL, ",", &pchr)) {
				// Parse each Bridge path and validate:

				bbfdm_get_reference_linker(ctx, pch, &reference);

				if (dm_validate_allowed_objects(ctx, &reference, allowed_objects))
					return FAULT_9007;
			}

			break;
		case VALUESET:
			for (pch = strtok_r(buf, ",", &pchr); pch != NULL; pch = strtok_r(NULL, ",", &pchr)) {

				bbfdm_get_reference_linker(ctx, pch, &reference);

				set_Provider_bridge_component(refparam, ctx, data, instance, reference.value, "CVLAN");
			}
			break;
	}
	return 0;
}

/**********************************************************************************************************************************
*                                            OBJ & PARAM DEFINITION
***********************************************************************************************************************************/
/*** Bridging.Bridge.{i}.STP. ***/
DMLEAF tBridgingBridgeSTPParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Enable", &DMWRITE, DMT_BOOL, get_BridgingBridgeSTP_Enable, set_BridgingBridgeSTP_Enable, BBFDM_BOTH},
{"Status", &DMREAD, DMT_STRING, get_BridgingBridgeSTP_Status, NULL, BBFDM_BOTH},
{"Protocol", &DMWRITE, DMT_STRING, get_BridgingBridgeSTP_Protocol, set_BridgingBridgeSTP_Protocol, BBFDM_BOTH},
{"BridgePriority", &DMWRITE, DMT_UNINT, get_BridgingBridgeSTP_BridgePriority, set_BridgingBridgeSTP_BridgePriority, BBFDM_BOTH},
{"HelloTime", &DMWRITE, DMT_UNINT, get_BridgingBridgeSTP_HelloTime, set_BridgingBridgeSTP_HelloTime, BBFDM_BOTH},
{"MaxAge", &DMWRITE, DMT_UNINT, get_BridgingBridgeSTP_MaxAge, set_BridgingBridgeSTP_MaxAge, BBFDM_BOTH},
{"ForwardingDelay", &DMWRITE, DMT_UNINT, get_BridgingBridgeSTP_ForwardingDelay, set_BridgingBridgeSTP_ForwardingDelay, BBFDM_BOTH},
{0}
};

/*** Bridging.Bridge.{i}.VLAN.{i}. ***/
DMLEAF tBridgingBridgeVLANParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Enable", &DMWRITE, DMT_BOOL, get_BridgingBridgeVLAN_Enable, set_BridgingBridgeVLAN_Enable, BBFDM_BOTH},
{"Alias", &DMWRITE, DMT_STRING, get_BridgingBridgeVLAN_Alias, set_BridgingBridgeVLAN_Alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Name", &DMWRITE, DMT_STRING, get_BridgingBridgeVLAN_Name, set_BridgingBridgeVLAN_Name, BBFDM_BOTH},
{"VLANID", &DMWRITE, DMT_INT, get_BridgingBridgeVLAN_VLANID, set_BridgingBridgeVLAN_VLANID, BBFDM_BOTH, DM_FLAG_UNIQUE|DM_FLAG_LINKER},
{0}
};

/*** Bridging.Bridge.{i}.VLANPort.{i}. ***/
DMLEAF tBridgingBridgeVLANPortParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Enable", &DMWRITE, DMT_BOOL, get_BridgingBridgeVLANPort_Enable, set_BridgingBridgeVLANPort_Enable, BBFDM_BOTH},
{"Alias", &DMWRITE, DMT_STRING,  get_BridgingBridgeVLANPort_Alias, set_BridgingBridgeVLANPort_Alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"VLAN", &DMWRITE, DMT_STRING,  get_BridgingBridgeVLANPort_VLAN, set_BridgingBridgeVLANPort_VLAN, BBFDM_BOTH, DM_FLAG_REFERENCE|DM_FLAG_UNIQUE},
{"Port", &DMWRITE, DMT_STRING, get_BridgingBridgeVLANPort_Port, set_BridgingBridgeVLANPort_Port, BBFDM_BOTH, DM_FLAG_REFERENCE|DM_FLAG_UNIQUE},
{"Untagged", &DMWRITE, DMT_BOOL, get_BridgingBridgeVLANPort_Untagged, set_BridgingBridgeVLANPort_Untagged, BBFDM_BOTH},
{0}
};

/*** Bridging.ProviderBridge.{i}. ***/
DMLEAF tBridgingProviderBridgeParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Enable", &DMWRITE, DMT_BOOL, get_BridgingBridgeProviderBridge_Enable, set_BridgingBridgeProviderBridge_Enable, BBFDM_BOTH},
{"Status", &DMREAD, DMT_STRING, get_BridgingBridgeProviderBridge_Status, NULL, BBFDM_BOTH},
{"Alias", &DMWRITE, DMT_STRING, get_BridgingBridgeProviderBridge_Alias, set_BridgingBridgeProviderBridge_Alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"SVLANcomponent", &DMWRITE, DMT_STRING, get_BridgingBridgeProviderBridge_SVLANcomponent, set_BridgingBridgeProviderBridge_SVLANcomponent, BBFDM_BOTH, DM_FLAG_REFERENCE},
{"CVLANcomponents", &DMWRITE, DMT_STRING, get_BridgingBridgeProviderBridge_CVLANcomponents, set_BridgingBridgeProviderBridge_CVLANcomponents, BBFDM_BOTH, DM_FLAG_REFERENCE},
{0}
};

/*** Bridging.Bridge.{i}.Port.{i}.Stats. ***/
DMLEAF tBridgingBridgePortStatsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"BytesSent", &DMREAD, DMT_UNLONG, get_BridgingBridgePortStats_BytesSent, NULL, BBFDM_BOTH},
{"BytesReceived", &DMREAD, DMT_UNLONG, get_BridgingBridgePortStats_BytesReceived, NULL, BBFDM_BOTH},
{"PacketsSent", &DMREAD, DMT_UNLONG, get_BridgingBridgePortStats_PacketsSent, NULL, BBFDM_BOTH},
{"PacketsReceived", &DMREAD, DMT_UNLONG, get_BridgingBridgePortStats_PacketsReceived, NULL, BBFDM_BOTH},
{"ErrorsSent", &DMREAD, DMT_UNINT, get_BridgingBridgePortStats_ErrorsSent, NULL, BBFDM_BOTH},
{"ErrorsReceived", &DMREAD, DMT_UNINT, get_BridgingBridgePortStats_ErrorsReceived, NULL, BBFDM_BOTH},
{"UnicastPacketsSent", &DMREAD, DMT_UNLONG, get_BridgingBridgePortStats_UnicastPacketsSent, NULL, BBFDM_BOTH},
{"UnicastPacketsReceived", &DMREAD, DMT_UNLONG, get_BridgingBridgePortStats_UnicastPacketsReceived, NULL, BBFDM_BOTH},
{"DiscardPacketsSent", &DMREAD, DMT_UNINT, get_BridgingBridgePortStats_DiscardPacketsSent, NULL, BBFDM_BOTH},
{"DiscardPacketsReceived", &DMREAD, DMT_UNINT, get_BridgingBridgePortStats_DiscardPacketsReceived, NULL, BBFDM_BOTH},
{"MulticastPacketsSent", &DMREAD, DMT_UNLONG, get_BridgingBridgePortStats_MulticastPacketsSent, NULL, BBFDM_BOTH},
{"MulticastPacketsReceived", &DMREAD, DMT_UNLONG, get_BridgingBridgePortStats_MulticastPacketsReceived, NULL, BBFDM_BOTH},
{"BroadcastPacketsSent", &DMREAD, DMT_UNLONG, get_BridgingBridgePortStats_BroadcastPacketsSent, NULL, BBFDM_BOTH},
{"BroadcastPacketsReceived", &DMREAD, DMT_UNLONG, get_BridgingBridgePortStats_BroadcastPacketsReceived, NULL, BBFDM_BOTH},
{"UnknownProtoPacketsReceived", &DMREAD, DMT_UNINT, get_BridgingBridgePortStats_UnknownProtoPacketsReceived, NULL, BBFDM_BOTH},
{0}
};

/*** Bridging.Bridge.{i}.Port.{i}. ***/
DMOBJ tBridgingBridgePortObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Stats", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tBridgingBridgePortStatsParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tBridgingBridgePortParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Enable", &DMWRITE, DMT_BOOL, get_BridgingBridgePort_Enable, set_BridgingBridgePort_Enable, BBFDM_BOTH},
{"Status", &DMREAD, DMT_STRING, get_BridgingBridgePort_Status, NULL, BBFDM_BOTH},
{"Alias", &DMWRITE, DMT_STRING, get_BridgingBridgePort_Alias, set_BridgingBridgePort_Alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Name", &DMREAD, DMT_STRING, get_BridgingBridgePort_Name, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE|DM_FLAG_LINKER},
//{"LastChange", &DMREAD, DMT_UNINT, get_BridgingBridgePort_LastChange, NULL, BBFDM_BOTH},
{"LowerLayers", &DMWRITE, DMT_STRING, get_BridgingBridgePort_LowerLayers, set_BridgingBridgePort_LowerLayers, BBFDM_BOTH, DM_FLAG_REFERENCE},
{"ManagementPort", &DMWRITE, DMT_BOOL, get_BridgingBridgePort_ManagementPort, set_BridgingBridgePort_ManagementPort, BBFDM_BOTH},
//{"Type", &DMWRITE, DMT_STRING, get_BridgingBridgePort_Type, set_BridgingBridgePort_Type, BBFDM_BOTH},
//{"DefaultUserPriority", &DMWRITE, DMT_UNINT, get_BridgingBridgePort_DefaultUserPriority, set_BridgingBridgePort_DefaultUserPriority, BBFDM_BOTH},
//{"PriorityRegeneration", &DMWRITE, DMT_STRING, get_BridgingBridgePort_PriorityRegeneration, set_BridgingBridgePort_PriorityRegeneration, BBFDM_BOTH},
//{"PortState", &DMREAD, DMT_STRING, get_BridgingBridgePort_PortState, NULL, BBFDM_BOTH},
{"PVID", &DMWRITE, DMT_INT, get_BridgingBridgePort_PVID, set_BridgingBridgePort_PVID, BBFDM_BOTH},
{"TPID", &DMWRITE, DMT_UNINT, get_BridgingBridgePort_TPID, set_BridgingBridgePort_TPID, BBFDM_BOTH},
{0}
};

/*** Bridging.Bridge.{i}. ***/
DMOBJ tBridgingBridgeObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"STP", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tBridgingBridgeSTPParams, NULL, BBFDM_BOTH},
{"Port", &DMWRITE, addObjBridgingBridgePort, delObjBridgingBridgePort, NULL, browseBridgingBridgePortInst, NULL, NULL, tBridgingBridgePortObj, tBridgingBridgePortParams, NULL, BBFDM_BOTH, NULL},
{"VLAN", &DMWRITE, addObjBridgingBridgeVLAN, delObjBridgingBridgeVLAN, NULL, browseBridgingBridgeVLANInst, NULL, NULL, NULL, tBridgingBridgeVLANParams, NULL, BBFDM_BOTH, NULL},
{"VLANPort", &DMWRITE, addObjBridgingBridgeVLANPort, delObjBridgingBridgeVLANPort, NULL, browseBridgingBridgeVLANPortInst, NULL, NULL, NULL, tBridgingBridgeVLANPortParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tBridgingBridgeParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Enable", &DMWRITE, DMT_BOOL, get_BridgingBridge_Enable, set_BridgingBridge_Enable, BBFDM_BOTH},
{"Status", &DMREAD, DMT_STRING, get_BridgingBridge_Status, NULL, BBFDM_BOTH},
{"Alias", &DMWRITE, DMT_STRING, get_BridgingBridge_Alias, set_BridgingBridge_Alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Name", &DMREAD, DMT_STRING, get_BridgingBridge_Name, NULL, BBFDM_BOTH, DM_FLAG_LINKER},
{"Standard", &DMWRITE, DMT_STRING, get_BridgingBridge_Standard, set_BridgingBridge_Standard, BBFDM_BOTH},
{"PortNumberOfEntries", &DMREAD, DMT_UNINT, get_BridgingBridge_PortNumberOfEntries, NULL, BBFDM_BOTH},
{"VLANNumberOfEntries", &DMREAD, DMT_UNINT, get_BridgingBridge_VLANNumberOfEntries, NULL, BBFDM_BOTH},
{"VLANPortNumberOfEntries", &DMREAD, DMT_UNINT, get_BridgingBridge_VLANPortNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.Bridging. *** */
DMLEAF tBridgingParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"MaxBridgeEntries", &DMREAD, DMT_UNINT, get_Bridging_MaxBridgeEntries, NULL, BBFDM_BOTH},
{"MaxDBridgeEntries", &DMREAD, DMT_UNINT, get_Bridging_MaxDBridgeEntries, NULL, BBFDM_BOTH},
{"MaxQBridgeEntries", &DMREAD, DMT_UNINT, get_Bridging_MaxQBridgeEntries, NULL, BBFDM_BOTH},
{"MaxVLANEntries", &DMREAD, DMT_UNINT, get_Bridging_MaxVLANEntries, NULL, BBFDM_BOTH},
{"MaxProviderBridgeEntries", &DMREAD, DMT_UNINT, get_Bridging_MaxProviderBridgeEntries, NULL, BBFDM_BOTH},
{"ProviderBridgeNumberOfEntries", &DMREAD, DMT_UNINT, get_Bridging_ProviderBridgeNumberOfEntries, NULL, BBFDM_BOTH},
{"MaxFilterEntries", &DMREAD, DMT_UNINT, get_Bridging_get_Bridging_MaxFilterEntries, NULL, BBFDM_BOTH},
{"BridgeNumberOfEntries", &DMREAD, DMT_UNINT, get_Bridging_BridgeNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

DMOBJ tBridgingObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Bridge", &DMWRITE, addObjBridgingBridge, delObjBridgingBridge, NULL, browseBridgingBridgeInst, NULL, NULL, tBridgingBridgeObj, tBridgingBridgeParams, NULL, BBFDM_BOTH, NULL},
{"ProviderBridge", &DMWRITE, addObjBridgingProviderBridge, delObjBridgingProviderBridge, NULL, browseBridgingProviderBridgeInst, NULL, NULL, NULL, tBridgingProviderBridgeParams, NULL, BBFDM_BOTH, NULL},
{0}
};

/*** Device. ***/
DMOBJ tDeviceObjs[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys*/
{"Bridging", &DMREAD, NULL, NULL, "file:/etc/config/network", NULL, NULL, NULL, tBridgingObj, tBridgingParams, NULL, BBFDM_BOTH, NULL},
{0}
};

/* ********** DynamicObj ********** */
DM_MAP_OBJ tDynamicObj[] = {
/* parentobj, nextobject, parameter */
{"Device.", tDeviceObjs, NULL},
{0}
};

